/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package org.openzen.entitysyncer.structured;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.openzen.entitysyncer.datasets.DatasetSubscriber;
import org.openzen.entitysyncer.datasets.DatasetSubscription;
import org.openzen.entitysyncer.structured.commit.StructuredCommitHistoryItem;
import org.openzen.packetstreams.ByteString;
import org.openzen.packetstreams.io.BytesDataOutput;

/**
 *
 * @author Hoofdgebruiker
 */
public class StructuredMemoryDataset implements StructuredDataset {
	private final StructuredClassRegistry registry;
	public final StructuredObject root;
	private final List<Subscription> subscriptions = new ArrayList<>();
	private final List<StructuredObject> objects = new ArrayList<>();
	private final Map<StructuredObject, Integer> idsByObject = new HashMap<>();
	private final Map<ByteString, StructuredCommitHistoryItem> commitHistory = new HashMap<>();
	
	private byte[] currentVersion;
	
	public StructuredMemoryDataset(StructuredClassRegistry registry, StructuredObject root, byte[] currentVersion) {
		this.registry = registry;
		this.root = root;
		this.currentVersion = currentVersion;
	}

	@Override
	public DatasetSubscription connect(DatasetSubscriber subscriber) {
		synchronized (subscriptions) {
			subscriber.onNewState(currentVersion, serializeState());
			Subscription subscription = new Subscription(subscriber);
			subscriptions.add(subscription);
			return subscription;
		}
	}

	@Override
	public DatasetSubscription connect(byte[] oldVersion, DatasetSubscriber subscriber) {
		synchronized (subscriptions) {
			// TODO: improve performance! This will block everything until all operations have been pushed
			List<StructuredCommitHistoryItem> history = getHistorySince(oldVersion);
			if (history == null)
				return connect(subscriber);
			
			for (StructuredCommitHistoryItem operation : history) {
				subscriber.onUpdated(operation.id.bytes(), operation.commit);
			}
			
			Subscription subscription = new Subscription(subscriber);
			subscriptions.add(subscription);
			return subscription;
		}
	}
	
	private List<StructuredCommitHistoryItem> getHistorySince(byte[] oldVersion) {
		ByteString oldVersionByteString = new ByteString(oldVersion);
		if (!commitHistory.containsKey(oldVersionByteString))
			return null;
		
		StructuredCommitHistoryItem item = commitHistory.get(oldVersionByteString);
		if (item.getNext() == null)
			return Collections.emptyList();
		
		List<StructuredCommitHistoryItem> result = new ArrayList<>();
		item = item.getNext();
		while (item != null) {
			result.add(item);
			item = item.getNext();
		}
		return result;
	}
	
	private void publish(StructuredCommitHistoryItem operation) {
		synchronized (subscriptions) {
			for (Subscription subscription : subscriptions) {
				subscription.subscriber.onUpdated(operation.id.bytes(), operation.commit);
			}
		}
	}
	
	private byte[] serializeState() {
		BytesDataOutput output = new BytesDataOutput();
		StructuredDataOutputStream scope = new StructuredDataOutputStream(registry, this, output);
		output.writeVarUInt(objects.size());
		for (StructuredObject object : objects) {
			scope.write(object);
		}
		return output.toByteArray();
	}

	@Override
	public StructuredObject getObject(int id) {
		return id < 0 || id >= objects.size() ? null : objects.get(id);
	}

	@Override
	public int getId(StructuredObject object) {
		Integer result = idsByObject.get(object);
		return result == null ? -1 : result;
	}

	@Override
	public int addObject(StructuredObject object) {
		int id = objects.size();
		objects.add(object);
		idsByObject.put(object, id);
		return id;
	}
	
	private class Subscription implements DatasetSubscription {
		private final DatasetSubscriber subscriber;
		
		public Subscription(DatasetSubscriber subscriber) {
			this.subscriber = subscriber;
		}
		
		@Override
		public void close() {
			synchronized (subscriptions) {
				subscriptions.remove(this);
			}
		}
	}
}
