/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package org.openzen.entitysyncer.structured;

import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import org.openzen.entitysyncer.structured.definition.StructFieldType;
import org.openzen.entitysyncer.structured.definition.StructuredObjectClass;

/**
 *
 * @author Hoofdgebruiker
 */
public class StructuredClassRegistry {
	private final Map<UUID, StructuredClassEntry<?>> entriesByUUID = new HashMap<>();
	private final Map<Class<?>, StructuredClassEntry<?>> entriesByClass = new HashMap<>();
	private final Map<UUID, StructFieldType<?>> structTypes = new HashMap<>();
	
	public <T extends StructuredObject> void register(Class<T> cls, StructuredObjectClass objectClass, StructuredClassFactory<T> factory) {
		if (objectClass.superclass != null) {
			StructuredClassEntry<?> superclassEntry = entriesByUUID.get(objectClass.superclass.id);
			if (superclassEntry == null)
				throw new IllegalArgumentException("Superclass is not yet registered! Make sure to register the superclass first.");
		}
		
		StructuredClassEntry<T> entry = new StructuredClassEntry<>(objectClass.id, cls, factory, objectClass);
		entriesByUUID.put(objectClass.id, entry);
		entriesByClass.put(cls, entry);
	}
	
	public void register(StructuredObjectClass objectClass) {
		StructuredClassEntry<?> superclassEntry = null;
		if (objectClass.superclass != null) {
			superclassEntry = entriesByUUID.get(objectClass.superclass.id);
			if (superclassEntry == null)
				throw new IllegalArgumentException("Superclass is not yet registered! Make sure to register the superclass first.");
		}
		
		StructuredClassEntry entry = new StructuredClassEntry<>(objectClass.id, null, new GenericFactory(superclassEntry, objectClass), objectClass);
		entriesByUUID.put(objectClass.id, entry);
	}
	
	public void register(StructFieldType<?> type) {
		structTypes.put(type.id, type);
	}
	
	public boolean knows(UUID uuid) {
		return !entriesByUUID.containsKey(uuid);
	}
	
	public StructuredObject read(UUID clsid, StructuredDataInputStream input) {
		return entriesByUUID.get(clsid).factory.read(input);
	}
	
	public StructFieldType<?> getStructType(UUID id) {
		return structTypes.get(id);
	}
	
	public StructuredObjectClass getStructuredClass(UUID id) {
		return entriesByUUID.get(id).structuredClass;
	}
	
	private class StructuredClassEntry<T extends StructuredObject> {
		public final UUID id;
		public final Class<T> cls;
		public final StructuredClassFactory<T> factory;
		public final StructuredObjectClass structuredClass;
		
		private StructuredClassEntry(UUID id, Class<T> cls, StructuredClassFactory<T> factory, StructuredObjectClass structuredClass) {
			this.id = id;
			this.cls = cls;
			this.factory = factory;
			this.structuredClass = structuredClass;
		}
	}
	
	private class GenericFactory implements StructuredClassFactory<BaseStructuredObject> {
		private final StructuredClassEntry<?> superEntry;
		private final StructuredObjectClass definition;
		
		public GenericFactory(StructuredClassEntry<?> superEntry, StructuredObjectClass definition) {
			this.superEntry = superEntry;
			this.definition = definition;
		}

		@Override
		public BaseStructuredObject read(StructuredDataInputStream input) {
			return new GenericStructuredObject(definition, input);
		}
	}
}
