/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package org.openzen.entitysyncer.datasets.stored;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.openzen.entitysyncer.datasets.Dataset;
import org.openzen.entitysyncer.datasets.DatasetSubscriber;
import org.openzen.entitysyncer.datasets.DatasetSubscription;
import org.openzen.entitysyncer.structured.StructuredClassRegistry;
import org.openzen.entitysyncer.structured.StructuredDataOutputStream;
import org.openzen.entitysyncer.structured.StructuredObject;
import org.openzen.packetstreams.ByteString;
import org.openzen.packetstreams.io.BytesDataInput;
import org.openzen.packetstreams.io.BytesDataOutput;
import org.openzen.packetstreams.io.DataOutput;
import org.openzen.packetstreams.io.RuntimeIOException;

/**
 *
 * @author Hoofdgebruiker
 */
public class StoredDataset implements Dataset {
	private static final int INDEX_SIZE = 16 * 1024;
	
	private final RandomAccessFile randomAccessFile;
	private final DeltaDataset dataset;
	private final StructuredClassRegistry registry;
	private final Map<ByteString, BlockIndex> index = new HashMap<>();
	private IndexBlock lastIndexBlock;
	private BlockIndex lastBlock;
	
	private final List<StoredDatasetSubscription> subscriptions = new ArrayList<>();
	
	public StoredDataset(File file, DeltaDataset dataset, StructuredClassRegistry registry) throws IOException {
		this.dataset = dataset;
		this.registry = registry;
		randomAccessFile = new RandomAccessFile(file, "rw");
		
		if (file.exists()) {
			lastIndexBlock = new IndexBlock(0, INDEX_SIZE);
			lastIndexBlock.writeHeader(randomAccessFile);

			byte[] initialData = serializeState();
			BytesDataOutput version = new BytesDataOutput();
			version.writeVarULong(dataset.getVersion());
			BlockIndex initialDataIndex = new BlockIndex(randomAccessFile.length(), initialData.length, version.toByteArray());
			addBlockIndex(initialDataIndex);
			lastBlock = initialDataIndex;

			randomAccessFile.seek(randomAccessFile.length());
			randomAccessFile.write(initialData);
		} else {
			IndexBlock indexBlock = new IndexBlock(0, randomAccessFile);
			while (true) {
				for (BlockIndex index : indexBlock.indexes)
					this.index.put(new ByteString(index.version), index);

				if (indexBlock.nextOffset == 0)
					break;

				indexBlock = new IndexBlock(indexBlock.nextOffset, randomAccessFile);

				for (BlockIndex block : indexBlock.indexes) {
					if (lastBlock != null)
						lastBlock.next = block;

					lastBlock = block;
				}
			}
		}
	}
	
	public void publish(byte[] version, byte[] delta) {
		try {
			BlockIndex newBlock = new BlockIndex(randomAccessFile.length(), delta.length, version);
			addBlockIndex(newBlock);

			randomAccessFile.seek(randomAccessFile.length());
			randomAccessFile.write(delta);
			
			lastBlock.next = newBlock;
			lastBlock = newBlock;
			
			synchronized (subscriptions) {
				for (StoredDatasetSubscription subscription : subscriptions) {
					subscription.subscriber.onUpdated(version, delta);
				}
			}
		} catch (IOException ex) {
			throw new RuntimeIOException(ex);
		}
	}
	
	@Override
	public DatasetSubscription connect(DatasetSubscriber subscriber) {
		byte[] state = serializeState();
		subscriber.onNewState(lastBlock.version, state);
		
		synchronized(subscriptions) {
			StoredDatasetSubscription subscription = new StoredDatasetSubscription(subscriber);
			subscriptions.add(subscription);
			return subscription;
		}
	}
	
	@Override
	public DatasetSubscription connect(byte[] oldVersion, DatasetSubscriber subscriber) {
		BlockIndex versionIndex = index.get(new ByteString(oldVersion));
		if (versionIndex == null)
			return connect(subscriber);
		
		while (versionIndex != null) {
			try {
				byte[] data = versionIndex.read(randomAccessFile);
				subscriber.onUpdated(versionIndex.version, data);
				versionIndex = versionIndex.next;
			} catch (IOException ex) {
				throw new RuntimeException("Could not load versions!", ex);
			}
		}
		
		StoredDatasetSubscription subscription = new StoredDatasetSubscription(subscriber);
		subscriptions.add(subscription);
		return subscription;
	}

	private void addBlockIndex(BlockIndex index) throws IOException {
		this.index.put(new ByteString(index.version), index);
		
		BytesDataOutput output = new BytesDataOutput();
		index.write(output);
		byte[] encoded = output.toByteArray();
		if (lastIndexBlock.sizeInBytes + encoded.length > lastIndexBlock.capacityInBytes) {
			long startOfNewBlock = randomAccessFile.length();
			lastIndexBlock.writeNextBlockOffset(randomAccessFile, startOfNewBlock);
			lastIndexBlock = new IndexBlock(randomAccessFile.length(), INDEX_SIZE);
			lastIndexBlock.writeHeader(randomAccessFile);
		}
		
		lastIndexBlock.writeIndexBlock(randomAccessFile, encoded);
	}
	
	private byte[] serializeState() {
		List<StructuredObject> objects = dataset.getObjects();
		BytesDataOutput rawOutput = new BytesDataOutput();
		StructuredDataOutputStream output = new StructuredDataOutputStream(registry, rawOutput);
		
		output.output.writeVarUInt(objects.size());
		for (StructuredObject object : objects) {
			output.write(object);
		}
		
		return rawOutput.toByteArray();
	}
	
	private static class IndexBlock {
		private final List<BlockIndex> indexes = new ArrayList<>();
		private final long offset;
		private final int capacityInBytes;
		private long nextOffset;
		private int sizeInBytes;
		
		public IndexBlock(long offset, int capacityInBytes) {
			this.offset = offset;
			this.capacityInBytes = capacityInBytes;
			this.nextOffset = 0;
			this.sizeInBytes = 0;
		}
		
		public IndexBlock(long offset, RandomAccessFile file) throws IOException {
			this.offset = offset;
			
			file.seek(offset);
			capacityInBytes = file.readInt();
			byte[] data = new byte[capacityInBytes - 12];
			file.readFully(data);
			
			BytesDataInput input = new BytesDataInput(data);
			while (true) {
				long blockOffset = input.readVarULong();
				if (blockOffset == 0)
					break;
				
				int size = input.readVarUInt();
				byte[] version = input.readByteArray();
				indexes.add(new BlockIndex(blockOffset, size, version));
			}
		}
		
		public void add(BlockIndex index, int sizeInBytes) {
			indexes.add(index);
			this.sizeInBytes += sizeInBytes;
		}
		
		public void writeHeader(RandomAccessFile file) throws IOException {
			file.seek(offset);
			file.writeLong(nextOffset);
			file.writeInt(sizeInBytes);
			file.write(new byte[capacityInBytes - 12]); // write zeroes - important to know where the block list ends
		}
		
		public void writeIndexBlock(RandomAccessFile file, byte[] data) throws IOException {
			file.seek(offset + sizeInBytes);
			file.write(data);
			sizeInBytes += data.length;
		}
		
		public void writeNextBlockOffset(RandomAccessFile file, long nextOffset) throws IOException {
			this.nextOffset = nextOffset;
			
			file.seek(offset);
			file.writeLong(nextOffset);
		}
	}
	
	private static class BlockIndex {
		private long offset;
		private int size;
		private byte[] version;
		private BlockIndex next;
		
		public BlockIndex(long offset, int size, byte[] version) {
			this.offset = offset;
			this.size = size;
			this.version = version;
		}
		
		public byte[] read(RandomAccessFile data) throws IOException {
			data.seek(offset);
			
			byte[] result = new byte[size];
			data.readFully(result);
			return result;
		}
		
		public void write(DataOutput output) {
			output.writeVarULong(offset);
			output.writeVarUInt(size);
			output.writeByteArray(version);
		}
	}
	
	private class StoredDatasetSubscription implements DatasetSubscription {
		private final DatasetSubscriber subscriber;
		
		public StoredDatasetSubscription(DatasetSubscriber subscriber) {
			this.subscriber = subscriber;
		}
		
		@Override
		public void close() {
			synchronized (subscriptions) {
				subscriptions.remove(this);
			}
		}
	}
}
