/* MIT licensed - see LICENSE in the project root directory. */
package org.openzen.packetstreams.qpsp;

import java.util.TimerTask;
import org.openzen.packetstreams.NetworkLogger;
import org.openzen.packetstreams.io.BytesDataInput;
import org.openzen.packetstreams.io.BytesDataOutput;

/**
 * The control stream is a special stream for exchanging session control
 * messages.
 */
public class ControlStream extends QPSPStream {
	private KeepaliveTimerTask keepaliveTimer = null;
	private int keepaliveInterval = 20000;
	private int maxKeepaliveInterval = 120000;
	private long lastKeepalive = System.currentTimeMillis();
	
	public ControlStream(QPSPSession session, long localId, long remoteId) {
		super(session, localId, remoteId, null);
		
		keepaliveTimer = new KeepaliveTimerTask();
		timer.scheduleAtFixedRate(keepaliveTimer, 1000, 1000);
	}
	
	@Override
	public void handleUnknown(int type, BytesDataInput input) {
		switch (type) {
			case Constants.PACKET_INITIALIZED:
				long remoteFromStreamId = input.readVarULong();
				long remoteToStreamId = remoteFromStreamId + 2 * input.readVarUInt();
				int maxPacketSize = input.readVarUInt();
				int maxBufferSize = input.readVarUInt();
				logger.log(NetworkLogger.CATEGORY_FRAMES, localId, "<- INITIALIZED");
				handleInitialized(
						remoteFromStreamId,
						remoteToStreamId,
						maxPacketSize,
						maxBufferSize);
				return;
			case Constants.PACKET_KEEPALIVE:
				logger.log(NetworkLogger.CATEGORY_FRAMES, localId, "<- KEEPALIVE");
				long packetsReceived = input.readVarULong();
				return;
			default:
				super.handleUnknown(type, input);
		}
	}
	
	@Override
	public int getPriority() {
		return 100; // give this stream a high priority
	}
	
	private void handleInitialized(
			long remoteFromStreamId,
			long remoteToStreamId,
			int maxPacketSize,
			int maxBufferSize) {
		session.initialized(remoteFromStreamId, remoteToStreamId, maxPacketSize, maxBufferSize);
	}
	
	public void initialize(int keepaliveInterval, int maxKeepaliveInterval) {
		BytesDataOutput output = new BytesDataOutput();
		output.writeUByte(Constants.PACKET_INITIALIZED);
		output.writeVarULong(session.localFromStreamID);
		output.writeVarUInt((int)(session.localToStreamID - session.localFromStreamID) / 2);
		output.writeVarUInt(session.maxPacketSize);
		output.writeVarUInt(session.maxBufferSize);
		enqueueControlFrame(output.toByteArray());
		
		this.keepaliveInterval = keepaliveInterval;
		this.maxKeepaliveInterval = maxKeepaliveInterval;
	}
	
	private class KeepaliveTimerTask extends TimerTask {
		@Override
		public void run() {
			long now = System.currentTimeMillis();
			if ((now - session.getLastPacketSentTimestamp()) < keepaliveInterval
					&& (now - lastKeepalive) < maxKeepaliveInterval)
				return;
			
			lastKeepalive = now;
			BytesDataOutput output = new BytesDataOutput();
			output.writeVarUInt(Constants.PACKET_KEEPALIVE);
			output.writeVarULong(session.getTotalPacketsReceived());
			enqueueControlFrame(new RawPacket(output.toByteArray(), true, true));
		}
	}
}
