/* MIT licensed - see LICENSE in the project root directory. */
package org.openzen.packetstreams.qpsp.socket;

import java.io.IOException;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.SocketException;

/**
 * Uses an actual UDP socket as underlying implementation.
 */
public class PureUDPSocket implements UDPSocket {
	private final int port;
	private final DatagramPacket buffer = new DatagramPacket(new byte[65536], 65536);
	
	private DatagramSocket socket = null;
	private int totalPacketsTransmitted = 0;
	private long totalDataTransmitted = 0;
	private int totalPacketsReceived = 0;
	private long totalDataReceived = 0;
	
	public PureUDPSocket(int port) {
		this.port = port;
	}
	
	@Override
	public void open() throws SocketException {
		socket = new DatagramSocket(port);
	}

	@Override
	public DatagramPacket receive() throws IOException {
		if (socket == null)
			throw new IOException("socket closed");
		
		socket.receive(buffer);
		totalPacketsReceived++;
		totalDataReceived += buffer.getLength();
		return buffer;
	}

	@Override
	public void send(DatagramPacket packet) throws IOException {
		if (socket == null)
			return;
		
		totalPacketsTransmitted++;
		totalDataTransmitted += packet.getLength();
		socket.send(packet);
	}

	@Override
	public void close() {
		if (socket == null)
			return;
		
		socket.close();
		socket = null;
	}

	@Override
	public int getTotalPacketsTransmitted() {
		return totalPacketsTransmitted;
	}

	@Override
	public long getTotalDataTransmitted() {
		return totalDataTransmitted;
	}

	@Override
	public int getTotalPacketsReceived() {
		return totalPacketsReceived;
	}

	@Override
	public long getTotalDataReceived() {
		return totalDataReceived;
	}
}
