/*
 * Decompiled with CFR 0.152.
 */
package org.openzen.packetstreams.qpsp;

import java.io.IOException;
import java.net.InetAddress;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import org.openzen.packetstreams.NetworkLogger;
import org.openzen.packetstreams.Server;
import org.openzen.packetstreams.ServiceConnector;
import org.openzen.packetstreams.ServiceMeta;
import org.openzen.packetstreams.Session;
import org.openzen.packetstreams.crypto.CryptoDecryptionException;
import org.openzen.packetstreams.crypto.CryptoKeyPair;
import org.openzen.packetstreams.crypto.CryptoProvider;
import org.openzen.packetstreams.crypto.CryptoPublicKey;
import org.openzen.packetstreams.crypto.CryptoSharedKey;
import org.openzen.packetstreams.io.BytesDataInput;
import org.openzen.packetstreams.io.BytesDataOutput;
import org.openzen.packetstreams.qpsp.ControlStream;
import org.openzen.packetstreams.qpsp.PacketScheduler;
import org.openzen.packetstreams.qpsp.QPSPEndpoint;
import org.openzen.packetstreams.qpsp.QPSPStream;
import org.openzen.packetstreams.qpsp.StandardPacketScheduler;
import org.openzen.packetstreams.qpsp.StreamMultiplexer;
import org.openzen.packetstreams.qpsp.scheduler.StandardCongestionController;

public class QPSPSession
implements Session {
    private static final int MAX_DECRYPTION_ERRORS = 16;
    private final QPSPEndpoint endpoint;
    public final Server server;
    public final NetworkLogger logger;
    public InetAddress remoteAddress;
    public int remotePort;
    public final long localFromStreamID;
    public final long localToStreamID;
    public long remoteFromStreamID;
    public long remoteToStreamID;
    public int maxPacketSize;
    public int maxBufferSize;
    public final int maxUDPPacketSize = 1200;
    public final long remoteNonce;
    public final long localNonce;
    private final CryptoPublicKey remotePublicKey;
    private final CryptoKeyPair keyPair;
    private final CryptoSharedKey key;
    private final byte[] outgoingNonce;
    private final byte[] incomingNonce;
    private final Map<Long, QPSPStream> streams = new HashMap<Long, QPSPStream>();
    private final ControlStream controlStream;
    private final StreamMultiplexer multiplexer;
    private final PacketScheduler scheduler;
    private final Thread commandThread;
    public final BlockingQueue<Runnable> commandQueue = new LinkedBlockingQueue<Runnable>();
    private long remoteStreamCounter;
    private boolean closed = false;
    private long totalPacketsReceived = 0L;
    private int decryptionErrors = 0;

    public QPSPSession(QPSPEndpoint endpoint, InetAddress address, int port, long localFromStreamID, long localToStreamID, int maxPacketSize, int maxBufferSize, long remoteNonce, long localNonce, Server server, CryptoPublicKey remotePublicKey, CryptoKeyPair keyPair) {
        this.endpoint = endpoint;
        this.server = server;
        this.logger = endpoint.logger;
        this.remoteAddress = address;
        this.remotePort = port;
        this.localFromStreamID = localFromStreamID;
        this.localToStreamID = localToStreamID;
        this.remoteFromStreamID = -1L;
        this.remoteToStreamID = -1L;
        this.remoteNonce = remoteNonce;
        this.localNonce = localNonce;
        this.remotePublicKey = remotePublicKey;
        this.keyPair = keyPair;
        this.key = endpoint.crypto.createSharedKey(remotePublicKey, keyPair.privateKey);
        this.multiplexer = new StreamMultiplexer(this);
        this.maxPacketSize = maxPacketSize;
        this.maxBufferSize = maxBufferSize;
        this.outgoingNonce = new byte[24];
        this.incomingNonce = new byte[24];
        QPSPSession.setLong(this.outgoingNonce, 0, remoteNonce);
        QPSPSession.setLong(this.incomingNonce, 0, localNonce);
        this.controlStream = new ControlStream(this, localFromStreamID, -1L);
        this.streams.put(localFromStreamID, this.controlStream);
        this.commandThread = new Thread(() -> {
            while (!this.closed) {
                try {
                    Runnable command = this.commandQueue.take();
                    if (command == null) continue;
                    command.run();
                }
                catch (InterruptedException interruptedException) {}
            }
        });
        this.commandThread.start();
        this.scheduler = new StandardPacketScheduler(this, this.multiplexer, endpoint, new StandardCongestionController());
    }

    @Override
    public CryptoPublicKey getRemoteKey() {
        return this.remotePublicKey;
    }

    @Override
    public void open(String path, ServiceConnector connector) {
        this.commandQueue.offer(() -> {
            QPSPStream stream = this.open(connector);
            stream.open(path, false);
            this.resume(stream);
        });
    }

    @Override
    public void open(String path, ServiceMeta cached, ServiceConnector connector) {
        this.commandQueue.offer(() -> {
            QPSPStream stream = this.open(connector);
            stream.open(path, true);
            stream.connect(cached);
            this.resume(stream);
        });
    }

    public CryptoProvider getCrypto() {
        return this.endpoint.crypto;
    }

    public int getEstimatedRTTInMillis() {
        return this.scheduler.getEstimatedRTTInMillis();
    }

    public long getTotalPacketsReceived() {
        return this.totalPacketsReceived;
    }

    public long getLastPacketSentTimestamp() {
        return this.scheduler.getLastSentPacketTimestamp();
    }

    private QPSPStream open(ServiceConnector connector) {
        long remoteStreamId = this.remoteStreamCounter;
        long localStreamId = remoteStreamId - this.remoteFromStreamID + this.localFromStreamID;
        this.remoteStreamCounter += 4L;
        QPSPStream stream = new QPSPStream(this, localStreamId, remoteStreamId, connector);
        this.streams.put(localStreamId, stream);
        return stream;
    }

    public void pause() {
        this.commandQueue.offer(() -> this.scheduler.pause());
    }

    public void resume() {
        this.commandQueue.offer(() -> this.scheduler.resume());
    }

    public void close() {
        this.commandQueue.offer(() -> {
            this.closed = true;
            this.scheduler.onSessionClosed();
        });
    }

    public void assertOnNetworkThread() {
        if (Thread.currentThread() != this.commandThread) {
            throw new AssertionError();
        }
    }

    public long getRemoteStreamCounter() {
        return this.remoteStreamCounter;
    }

    public void initClient(long remoteFromStreamId, long remoteToStreamId, int keepaliveInterval, int maxKeepaliveInterval) {
        this.remoteFromStreamID = remoteFromStreamId;
        this.remoteToStreamID = remoteToStreamId;
        this.remoteStreamCounter = this.remoteFromStreamID + 4L;
        this.controlStream.initialize(keepaliveInterval, maxKeepaliveInterval);
        System.out.println("Init client session; remote " + remoteFromStreamId + ", local " + this.localFromStreamID);
        for (QPSPStream stream : this.streams.values()) {
            stream.remoteId = stream.localId - this.localFromStreamID + this.remoteFromStreamID;
        }
    }

    public void initServer(long remoteFromStreamId, long remoteToStreamId, int maxPacketSize, int maxBufferSize) {
        this.remoteFromStreamID = remoteFromStreamId;
        this.remoteToStreamID = remoteToStreamId;
        this.remoteStreamCounter = remoteFromStreamId + 4L;
        this.maxPacketSize = maxPacketSize;
        this.maxBufferSize = maxBufferSize;
        System.out.println("Init server session; remote " + remoteFromStreamId + ", local " + this.localFromStreamID);
        for (QPSPStream stream : this.streams.values()) {
            stream.remoteId = stream.localId - this.localFromStreamID + remoteFromStreamId;
        }
    }

    public void initFromStorage(long remoteFromStreamId, long remoteToStreamId, long remoteStreamCounter) {
        this.remoteFromStreamID = remoteFromStreamId;
        this.remoteToStreamID = remoteToStreamId;
        this.remoteStreamCounter = remoteStreamCounter;
        for (QPSPStream stream : this.streams.values()) {
            stream.remoteId = stream.localId - this.localFromStreamID + remoteFromStreamId;
        }
    }

    public void setRemote(InetAddress address, int port) {
        this.commandQueue.offer(() -> {
            this.remoteAddress = address;
            this.remotePort = port;
        });
    }

    public void sendInit() {
        this.commandQueue.offer(() -> {
            try {
                BytesDataOutput output = new BytesDataOutput();
                output.writeVarULong(1L);
                output.writeULong(this.remoteNonce);
                output.writeULong(this.localNonce);
                output.writeVarUInt(0);
                output.writeRawBytes(this.keyPair.publicKey.encode());
                BytesDataOutput encrypted = new BytesDataOutput();
                encrypted.writeVarUInt(0);
                this.server.getCertificate().serialize(encrypted);
                encrypted.writeVarULong(this.localFromStreamID);
                encrypted.writeVarULong((this.localToStreamID - this.localFromStreamID) / 4L);
                encrypted.writeVarUInt(this.server.getMaxPacketSize());
                encrypted.writeVarUInt(this.server.getMaxBufferSize());
                output.writeByteArray(this.encrypt(encrypted.toByteArray(), 0L, 0L));
                this.endpoint.send(this, output.toByteArray());
            }
            catch (IOException ex) {
                ex.printStackTrace();
            }
        });
    }

    public byte[] encrypt(byte[] data, long streamId, long seq) {
        return this.encrypt(data, 0, data.length, streamId, seq);
    }

    public byte[] encrypt(byte[] data, int offset, int length, long streamId, long seq) {
        this.assertOnNetworkThread();
        QPSPSession.setLong(this.outgoingNonce, 8, streamId);
        QPSPSession.setLong(this.outgoingNonce, 16, seq);
        return this.key.encrypt(this.outgoingNonce, data, offset, length);
    }

    public byte[] decrypt(byte[] data, int offset, int length, long streamId, long seq) throws CryptoDecryptionException {
        this.assertOnNetworkThread();
        QPSPSession.setLong(this.incomingNonce, 8, streamId);
        QPSPSession.setLong(this.incomingNonce, 16, seq);
        return this.key.decrypt(this.incomingNonce, data, offset, length);
    }

    public void resume(QPSPStream stream) {
        this.assertOnNetworkThread();
        this.multiplexer.resume(stream);
    }

    public void doResume() {
        if (this.closed) {
            return;
        }
        this.scheduler.resumeStreams();
    }

    public boolean equals(QPSPSession other) {
        return this.remoteAddress.equals(other.remoteAddress) && this.remotePort == other.remotePort;
    }

    public CryptoPublicKey getPublicKey() {
        return this.keyPair.publicKey;
    }

    public void onReceived(long streamId, BytesDataInput input) {
        this.commandQueue.offer(() -> this.doReceive(streamId, input));
    }

    private void doReceive(long streamId, BytesDataInput input) {
        ++this.totalPacketsReceived;
        QPSPStream stream = this.getLocalStream(streamId & 0xFFFFFFFFFFFFFFFDL);
        long seq = stream.decodeCompactedSEQ(input);
        try {
            byte[] decrypted = this.decrypt(input.getData(), input.getCurrentOffset(), input.getAvailable(), streamId, seq);
            stream.onReceived(seq, (streamId & 2L) == 2L, decrypted);
            this.scheduler.onPacketReceived();
        }
        catch (CryptoDecryptionException ex) {
            this.logger.log(2, streamId, "Decryption error for #" + seq);
            ++this.decryptionErrors;
            if (this.decryptionErrors > 16) {
                // empty if block
            }
        }
    }

    public void onAcknowledged(long streamId, long seq) {
        this.scheduler.onAcknowledged(streamId, seq);
    }

    public void onClosed(long streamId) {
        this.scheduler.onStreamClosed(streamId);
    }

    private QPSPStream getLocalStream(long streamId) {
        QPSPStream stream = this.streams.get(streamId);
        if (stream == null) {
            this.logger.log(1, streamId, "New stream created for stream " + streamId);
            long remoteStreamId = this.remoteFromStreamID == -1L ? -1L : streamId - this.localFromStreamID + this.remoteFromStreamID;
            stream = new QPSPStream(this, streamId, remoteStreamId, null);
            this.streams.put(streamId, stream);
        }
        return stream;
    }

    public static void setLong(byte[] data, int offset, long value) {
        data[offset + 0] = (byte)(value >> 56);
        data[offset + 1] = (byte)(value >> 48);
        data[offset + 2] = (byte)(value >> 40);
        data[offset + 3] = (byte)(value >> 32);
        data[offset + 4] = (byte)(value >> 24);
        data[offset + 5] = (byte)(value >> 16);
        data[offset + 6] = (byte)(value >> 8);
        data[offset + 7] = (byte)value;
    }
}

