/* MIT licensed - see LICENSE in the project root directory. */
package org.openzen.packetstreams.io;

import java.io.IOException;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.time.LocalDate;
import java.util.UUID;

/**
 * DataOutput wrapping an OutputStream.
 */
public class StreamingDataOutput implements DataOutput
{
	private static final int P6 = 1 << 6;
	private static final int P7 = 1 << 7;
	private static final int P13 = 1 << 13;
	private static final int P14 = 1 << 14;
	private static final int P20 = 1 << 20;
	private static final int P21 = 1 << 21;
	private static final int P27 = 1 << 27;
	private static final int P28 = 1 << 28;
	private static final long P34 = 1L << 34;
	private static final long P35 = 1L << 35;
	private static final long P41 = 1L << 41;
	private static final long P42 = 1L << 42;
	private static final long P48 = 1L << 48;
	private static final long P49 = 1L << 49;
	private static final long P55 = 1L << 55;
	private static final long P56 = 1L << 56;
	
	private final OutputStream output;

    public StreamingDataOutput(OutputStream output)
	{
		this.output = output;
	}
	
	// ==================================
	// === IDataOutput Implementation ===
	// ==================================
	
	@Override
	public void writeBoolean(boolean value)
	{
		writeUByte(value ? 1 : 0);
	}
	
	@Override
    public void writeSByte(byte value)
	{
		try {
			output.write(value);
		} catch (IOException ex) {
			throw new RuntimeIOException(ex);
		}
    }
	
	@Override
	public void writeUByte(int value)
	{
		writeSByte((byte) value);
	}

	@Override
    public void writeShort(short value)
	{
		try {
			output.write(value >>> 8);
			output.write(value);
		} catch (IOException ex) {
			throw new RuntimeIOException(ex);
		}
    }
	
	@Override
	public void writeUShort(int value)
	{
		writeShort((short) value);
	}

	@Override
    public void writeInt(int value)
	{
		try {
			output.write(value >>> 24);
			output.write(value >>> 16);
			output.write(value >>> 8);
			output.write(value);
		} catch (IOException ex) {
			throw new RuntimeIOException(ex);
		}
    }
	
	@Override
	public void writeUInt(int value)
	{
		writeInt(value);
	}

	@Override
    public void writeLong(long value)
	{
		try {
			output.write((byte)(value >>> 56));
			output.write((byte)(value >>> 48));
			output.write((byte)(value >>> 40));
			output.write((byte)(value >>> 32));
			output.write((byte)(value >>> 24));
			output.write((byte)(value >>> 16));
			output.write((byte)(value >>> 8));
			output.write((byte)(value));
		} catch (IOException ex) {
			throw new RuntimeIOException(ex);
		}
    }
	
	@Override
	public void writeULong(long value)
	{
		writeLong(value);
	}
	
	@Override
	public void writeVarInt(int value)
	{
		try {
			if (value >= -P6 && value < P6) {
				value += P6;
				output.write((byte) (value & 0x7F));
			} else if (value >= -P13 && value < P13) {
				value += P13;
				output.write((byte) ((value & 0x7F) | 0x80));
				output.write((byte) ((value >> 7) & 0x7F));
			} else if (value >= -P20 && value < P20) {
				value += P20;
				output.write((byte) ((value & 0x7F) | 0x80));
				output.write((byte) (((value >> 7) & 0x7F) | 0x80));
				output.write((byte) ((value >> 14) & 0x7F));
			} else if (value >= -P27 && value < P27) {
				value += P27;
				output.write((byte) ((value & 0x7F) | 0x80));
				output.write((byte) (((value >> 7) & 0x7F) | 0x80));
				output.write((byte) (((value >> 14) & 0x7F) | 0x80));
				output.write((byte) ((value >> 21) & 0x7F));
			} else {
				output.write((byte) ((value & 0x7F) | 0x80));
				output.write((byte) (((value >>> 7) & 0x7F) | 0x80));
				output.write((byte) (((value >>> 14) & 0x7F) | 0x80));
				output.write((byte) (((value >>> 21) & 0x7F) | 0x80));
				output.write((byte) ((value >>> 28) & 0x7F));
			}
		} catch (IOException ex) {
			throw new RuntimeIOException(ex);
		}
	}
	
	@Override
	public void writeVarUInt(int value)
	{
		try {
			if (value >= 0 && value < P28) {
				if ( value < P7) {
					output.write((byte) (value & 0x7F));
				} else if (value < P14) {
					output.write((byte) ((value & 0x7F) | 0x80));
					output.write((byte) ((value >> 7) & 0x7F));
				} else if (value < P21) {
					output.write((byte) ((value & 0x7F) | 0x80));
					output.write((byte) (((value >> 7) & 0x7F) | 0x80));
					output.write((byte) ((value >> 14) & 0x7F));
				} else {
					output.write((byte) ((value & 0x7F) | 0x80));
					output.write((byte) (((value >> 7) & 0x7F) | 0x80));
					output.write((byte) (((value >> 14) & 0x7F) | 0x80));
					output.write((byte) ((value >> 21) & 0x7F));
				}
			} else {
				output.write((byte) ((value & 0x7F) | 0x80));
				output.write((byte) (((value >>> 7) & 0x7F) | 0x80));
				output.write((byte) (((value >>> 14) & 0x7F) | 0x80));
				output.write((byte) (((value >>> 21) & 0x7F) | 0x80));
				output.write((byte) ((value >>> 28) & 0x7F));
			}
		} catch (IOException ex) {
			throw new RuntimeIOException(ex);
		}
	}
	
	@Override
	public void writeVarLong(long value)
	{
		try {
			if (value >= -P6 && value < P6) {
				value += P6;
				output.write((byte) (value & 0x7F));
			} else if (value >= -P13 && value < P13) {
				value += P13;
				output.write((byte) ((value & 0x7F) | 0x80));
				output.write((byte) ((value >> 7) & 0x7F));
			} else if (value >= -P20 && value < P20) {
				value += P20;
				output.write((byte) ((value & 0x7F) | 0x80));
				output.write((byte) (((value >> 7) & 0x7F) | 0x80));
				output.write((byte) ((value >> 14) & 0x7F));
			} else if (value >= -P27 && value < P27) {
				value += P27;
				output.write((byte) ((value & 0x7F) | 0x80));
				output.write((byte) (((value >> 7) & 0x7F) | 0x80));
				output.write((byte) (((value >> 14) & 0x7F) | 0x80));
				output.write((byte) ((value >> 21) & 0x7F));
			} else if (value >= -P34 && value < P34) {
				value += P34;
				output.write((byte) ((value & 0x7F) | 0x80));
				output.write((byte) (((value >> 7) & 0x7F) | 0x80));
				output.write((byte) (((value >> 14) & 0x7F) | 0x80));
				output.write((byte) (((value >> 21) & 0x7F) | 0x80));
				output.write((byte) ((value >> 28) & 0x7F));
			} else if (value >= -P41 && value < P42) {
				value += P41;
				output.write((byte) ((value & 0x7F) | 0x80));
				output.write((byte) (((value >> 7) & 0x7F) | 0x80));
				output.write((byte) (((value >> 14) & 0x7F) | 0x80));
				output.write((byte) (((value >> 21) & 0x7F) | 0x80));
				output.write((byte) (((value >> 28) & 0x7F) | 0x80));
				output.write((byte) ((value >> 35) & 0x7F));
			} else if (value >= -P48 && value < P48) {
				value += P48;
				output.write((byte) ((value & 0x7F) | 0x80));
				output.write((byte) (((value >> 7) & 0x7F) | 0x80));
				output.write((byte) (((value >> 14) & 0x7F) | 0x80));
				output.write((byte) (((value >> 21) & 0x7F) | 0x80));
				output.write((byte) (((value >> 28) & 0x7F) | 0x80));
				output.write((byte) (((value >> 35) & 0x7F) | 0x80));
				output.write((byte) ((value >> 42) & 0x7F));
			} else if (value >= -P55 && value < P55) {
				value += P55;
				output.write((byte) ((value & 0x7F) | 0x80));
				output.write((byte) (((value >> 7) & 0x7F) | 0x80));
				output.write((byte) (((value >> 14) & 0x7F) | 0x80));
				output.write((byte) (((value >> 21) & 0x7F) | 0x80));
				output.write((byte) (((value >> 28) & 0x7F) | 0x80));
				output.write((byte) (((value >> 35) & 0x7F) | 0x80));
				output.write((byte) (((value >> 42) & 0x7F) | 0x80));
				output.write((byte) ((value >> 49) & 0x7F));
			} else {
				output.write((byte) ((value & 0x7F) | 0x80));
				output.write((byte) (((value >>> 7) & 0x7F) | 0x80));
				output.write((byte) (((value >>> 14) & 0x7F) | 0x80));
				output.write((byte) (((value >>> 21) & 0x7F) | 0x80));
				output.write((byte) (((value >>> 28) & 0x7F) | 0x80));
				output.write((byte) (((value >>> 35) & 0x7F) | 0x80));
				output.write((byte) (((value >>> 42) & 0x7F) | 0x80));
				output.write((byte) (((value >>> 49) & 0x7F) | 0x80));
				output.write((byte) (value >>> 56));
			}
		} catch (IOException ex) {
			throw new RuntimeIOException(ex);
		}
	}
	
	@Override
	public void writeVarULong(long value)
	{
		try {
			if (value >= 0 && value < P56) {
				if ( value < P7) {
					output.write((byte) (value & 0x7F));
				} else if (value < P14) {
					output.write((byte) ((value & 0x7F) | 0x80));
					output.write((byte) ((value >> 7) & 0x7F));
				} else if (value < P21) {
					output.write((byte) ((value & 0x7F) | 0x80));
					output.write((byte) (((value >> 7) & 0x7F) | 0x80));
					output.write((byte) ((value >> 14) & 0x7F));
				} else if (value < P28) {
					output.write((byte) ((value & 0x7F) | 0x80));
					output.write((byte) (((value >> 7) & 0x7F) | 0x80));
					output.write((byte) (((value >> 14) & 0x7F) | 0x80));
					output.write((byte) ((value >> 21) & 0x7F));
				} else if (value < P35) {
					output.write((byte) ((value & 0x7F) | 0x80));
					output.write((byte) (((value >> 7) & 0x7F) | 0x80));
					output.write((byte) (((value >> 14) & 0x7F) | 0x80));
					output.write((byte) (((value >> 21) & 0x7F) | 0x80));
					output.write((byte) ((value >> 28) & 0x7F));
				} else if (value < P42) {
					output.write((byte) ((value & 0x7F) | 0x80));
					output.write((byte) (((value >> 7) & 0x7F) | 0x80));
					output.write((byte) (((value >> 14) & 0x7F) | 0x80));
					output.write((byte) (((value >> 21) & 0x7F) | 0x80));
					output.write((byte) (((value >> 28) & 0x7F) | 0x80));
					output.write((byte) ((value >> 35) & 0x7F));
				} else if (value < P49) {
					output.write((byte) ((value & 0x7F) | 0x80));
					output.write((byte) (((value >> 7) & 0x7F) | 0x80));
					output.write((byte) (((value >> 14) & 0x7F) | 0x80));
					output.write((byte) (((value >> 21) & 0x7F) | 0x80));
					output.write((byte) (((value >> 28) & 0x7F) | 0x80));
					output.write((byte) (((value >> 35) & 0x7F) | 0x80));
					output.write((byte) ((value >> 42) & 0x7F));
				} else {
					output.write((byte) ((value & 0x7F) | 0x80));
					output.write((byte) (((value >> 7) & 0x7F) | 0x80));
					output.write((byte) (((value >> 14) & 0x7F) | 0x80));
					output.write((byte) (((value >> 21) & 0x7F) | 0x80));
					output.write((byte) (((value >> 28) & 0x7F) | 0x80));
					output.write((byte) (((value >> 35) & 0x7F) | 0x80));
					output.write((byte) (((value >> 42) & 0x7F) | 0x80));
					output.write((byte) ((value >> 49) & 0x7F));
				}
			} else {
				output.write((byte) ((value & 0x7F) | 0x80));
				output.write((byte) (((value >>> 7) & 0x7F) | 0x80));
				output.write((byte) (((value >>> 14) & 0x7F) | 0x80));
				output.write((byte) (((value >>> 21) & 0x7F) | 0x80));
				output.write((byte) (((value >>> 28) & 0x7F) | 0x80));
				output.write((byte) (((value >>> 35) & 0x7F) | 0x80));
				output.write((byte) (((value >>> 42) & 0x7F) | 0x80));
				output.write((byte) (((value >>> 49) & 0x7F) | 0x80));
				output.write((byte) (value >>> 56));
			}
		} catch (IOException ex) {
			throw new RuntimeIOException(ex);
		}
	}
	
    @Override
	public void writeFloat(float value)
	{
        writeInt(Float.floatToIntBits(value));
    }
	
    @Override
	public void writeDouble(double value)
	{
        writeLong(Double.doubleToLongBits(value));
    }

	@Override
	public void writeChar(int value)
	{
		writeVarUInt(value);
	}
	
	@Override
    public void writeBytes(byte[] data)
	{
        writeVarUInt(data.length);
        writeRawBytes(data);
    }
	
	@Override
	public void writeBytes(byte[] data, int offset, int length)
	{
		writeVarUInt(length);
		writeRawBytes(data, offset, length);
	}
	
	@Override
	public void writeAllBytes(ByteBuffer value)
	{
		value.rewind();
		
		byte[] tmp = new byte[value.capacity()];
		value.get(tmp);
		writeBytes(tmp);
	}
	
	@Override
	public void writeBytes(ByteBuffer value, int length)
	{
		writeVarUInt(length);
		writeRawBytes(value, length);
	}
	
	@Override
	public void writeRawBytes(byte[] value)
	{
		try {
	        output.write(value);
		} catch (IOException ex) {
			throw new RuntimeIOException(ex);
		}
    }
	
	@Override
	public void writeRawBytes(byte[] value, int offset, int length)
	{
		try {
			output.write(value, offset, length);
		} catch (IOException ex) {
			throw new RuntimeIOException(ex);
		}
	}
	
	@Override
	public void writeRawBytes(ByteBuffer value, int length)
	{
		byte[] tmp = new byte[length];
		value.get(tmp, 0, length);
		writeRawBytes(tmp);
	}
	
	@Override
	public void writeByteArray(byte[] data)
	{
		writeBytes(data);
	}
	
	@Override
	public void writeUByteArray(byte[] data)
	{
		writeBytes(data);
	}
	
	@Override
	public void writeShortArray(short[] data)
	{
		writeVarUInt(data.length);
		for (short element : data)
			writeShort(element);
	}
	
	@Override
	public void writeShortArrayRaw(short[] data)
	{
		for (short element : data)
			writeShort(element);
	}
	
	@Override
	public void writeUShortArray(short[] data)
	{
		writeShortArray(data);
	}
	
	@Override
	public void writeUShortArrayRaw(short[] data)
	{
		writeShortArrayRaw(data);
	}
	
	@Override
	public void writeVarIntArray(int[] data)
	{
		writeVarUInt(data.length);
		for (int element : data)
			writeVarInt(element);
	}
	
	@Override
	public void writeVarIntArrayRaw(int[] data)
	{
		for (int element : data)
			writeVarInt(element);
	}
	
	@Override
	public void writeVarUIntArray(int[] data)
	{
		writeVarUInt(data.length);
		for (int element : data)
			writeVarUInt(element);
	}
	
	@Override
	public void writeVarUIntArrayRaw(int[] data)
	{
		for (int element : data)
			writeVarUInt(element);
	}
	
	@Override
	public void writeIntArray(int[] data)
	{
		writeVarUInt(data.length);
		for (int element : data)
			writeInt(element);
	}
	
	@Override
	public void writeIntArrayRaw(int[] data)
	{
		for (int element : data)
			writeInt(element);
	}
	
	@Override
	public void writeUIntArray(int[] data)
	{
		writeVarUInt(data.length);
		for (int element : data)
			writeUInt(element);
	}
	
	@Override
	public void writeUIntArrayRaw(int[] data)
	{
		for (int element : data)
			writeUInt(element);
	}
	
	@Override
	public void writeVarLongArray(long[] data)
	{
		writeVarUInt(data.length);
		for (long element : data)
			writeVarLong(element);
	}
	
	@Override
	public void writeVarLongArrayRaw(long[] data)
	{
		for (long element : data)
			writeVarLong(element);
	}
	
	@Override
	public void writeVarULongArray(long[] data)
	{
		writeVarUInt(data.length);
		for (long element : data)
			writeVarULong(element);
	}
	
	@Override
	public void writeVarULongArrayRaw(long[] data)
	{
		for (long element : data)
			writeVarULong(element);
	}
	
	@Override
	public void writeLongArray(long[] data)
	{
		writeVarUInt(data.length);
		for (long element : data)
			writeLong(element);
	}
	
	@Override
	public void writeLongArrayRaw(long[] data)
	{
		for (long element : data)
			writeLong(element);
	}
	
	@Override
	public void writeULongArray(long[] data)
	{
		writeVarUInt(data.length);
		for (long element : data)
			writeULong(element);
	}
	
	@Override
	public void writeULongArrayRaw(long[] data)
	{
		for (long element : data)
			writeULong(element);
	}
	
	@Override
	public void writeFloatArray(float[] data)
	{
		writeVarUInt(data.length);
		for (float element : data)
			writeFloat(element);
	}
	
	@Override
	public void writeFloatArrayRaw(float[] data)
	{
		for (float element : data)
			writeFloat(element);
	}
	
	@Override
	public void writeDoubleArray(double[] data)
	{
		writeVarUInt(data.length);
		for (double element : data)
			writeDouble(element);
	}
	
	@Override
	public void writeDoubleArrayRaw(double[] data)
	{
		for (double element : data)
			writeDouble(element);
	}
	
	@Override
	public void writeStringArray(String[] data)
	{
		writeVarUInt(data.length);
		for (String element : data)
			writeString(element);
	}
	
	@Override
	public void writeStringArrayRaw(String[] data)
	{
		for (String element : data)
			writeString(element);
	}
	
	@Override
    public void writeString(String str)
	{
		writeBytes(str.getBytes(StandardCharsets.UTF_8));
    }
	
	@Override
	public void writeDate(LocalDate value)
	{
		if (value == null) {
			writeVarInt(-32);
			return;
		}
		
		int ivalue = value.getYear() - 2000;
		ivalue = ivalue * 12 * 31
				+ (value.getMonthValue() - 1) * 31
				+ value.getDayOfMonth() - 1;
		writeVarInt(ivalue);
	}
	
	@Override
	public void writeUUID(UUID uuid) {
		writeLong(uuid.getMostSignificantBits());
		writeLong(uuid.getLeastSignificantBits());
	}
	
	@Override
	public void flush()
	{
		try {
			output.flush();
		} catch (IOException ex) {
			throw new RuntimeIOException(ex);
		}
	}

	@Override
	public void close()
	{
		try {
			output.close();
		} catch (IOException ex) {
			throw new RuntimeIOException(ex);
		}
	}
}
