/* MIT licensed - see LICENSE in the project root directory. */
package org.openzen.packetstreams.qpsp.socket;

import java.io.IOException;
import java.net.DatagramPacket;
import java.util.Random;
import java.util.Timer;
import java.util.TimerTask;

/**
 * Simulates conditions on a slow, lossy network.
 */
public class BadNetworkSimulator implements UDPSocket {
	private static final int PACKET_OVERHEAD = 20 + 8;
	
	private final UDPSocket actual;
	
	private final Random random = new Random(); // the bitch of mobile communication
	private final double lossRate;
	private final double duplicateProbability;
	private final int minDelay;
	private final int maxDelay;
	private final int bytesPerSecond;
	private final int networkBuffer;
	private final Timer timer = new Timer();
	
	private int networkBufferSize = 0;
	private long networkBufferTime = System.currentTimeMillis();
	private boolean closed = false;
	
	private int packetsTransmitted = 0;
	private long bytesTransmitted = 0;
	
	public BadNetworkSimulator(
			UDPSocket actual,
			double lossRate,
			double duplicateProbability,
			int minDelay,
			int maxDelay,
			int bytesPerSecond,
			int networkBuffer) {
		this.actual = actual;
		
		this.lossRate = lossRate;
		this.duplicateProbability = duplicateProbability;
		this.minDelay = minDelay;
		this.maxDelay = maxDelay;
		this.bytesPerSecond = bytesPerSecond;
		this.networkBuffer = networkBuffer + bytesPerSecond * (minDelay + maxDelay) / 1000;
	}

	@Override
	public DatagramPacket receive() throws IOException {
		return actual.receive();
	}

	@Override
	public void send(DatagramPacket packet) throws IOException {
		packetsTransmitted++;
		bytesTransmitted += packet.getLength();
		
		if (!tryToSendBytes(packet.getLength())) {
			System.out.println("Network overload, dropped packet");
			return;
		}
		if (random.nextDouble() < lossRate) {
			System.out.println("Dropped packet");
			return; // to the void!
		}
		
		int count = 0;
		do {
			if (count > 0)
				System.out.println("Duplicated packet");
			
			int delay = 1000 * (packet.getLength() + PACKET_OVERHEAD) / bytesPerSecond + minDelay + random.nextInt(maxDelay - minDelay);
			timer.schedule(new TransmissionTimerTask(packet), delay);
			count++;
		} while (random.nextDouble() < duplicateProbability);
	}

	@Override
	public void close() {
		closed = true;
		actual.close();
	}

	@Override
	public int getTotalPacketsTransmitted() {
		return packetsTransmitted;
	}

	@Override
	public long getTotalDataTransmitted() {
		return bytesTransmitted;
	}

	@Override
	public int getTotalPacketsReceived() {
		return actual.getTotalPacketsReceived();
	}

	@Override
	public long getTotalDataReceived() {
		return actual.getTotalDataReceived();
	}
	
	private boolean tryToSendBytes(int bytes) {
		long now = System.currentTimeMillis();
		int millis = (int)(now - networkBufferTime);
		int bytesInFlight = Math.max(0, networkBufferSize - millis * bytesPerSecond / 1000);
		if (bytesInFlight + bytes > networkBuffer)
			return false;
		
		networkBufferSize = bytesInFlight + bytes;
		networkBufferTime = now;
		return true;
	}
	
	private class TransmissionTimerTask extends TimerTask {
		private final DatagramPacket packet;

		public TransmissionTimerTask(DatagramPacket packet) {
			this.packet = packet;
		}
		
		@Override
		public void run() {
			if (closed)
				return;
			
			try {
				actual.send(packet);
			} catch (IOException ex) {
				ex.printStackTrace();
			}
		}
	}
}
