/* MIT licensed - see LICENSE in the project root directory. */
package org.openzen.packetstreams.io;

import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.time.LocalDate;
import java.util.UUID;

/**
 * DataInput implementation which wraps an InputStream.
 */
public class StreamingDataInput implements DataInput
{
	private static final int P6 = 1 << 6;
	private static final int P7 = 1 << 7;
	private static final int P13 = 1 << 13;
	private static final int P14 = 1 << 14;
	private static final int P20 = 1 << 20;
	private static final int P21 = 1 << 21;
	private static final int P27 = 1 << 27;
	private static final int P28 = 1 << 28;
	private static final long P34 = 1L << 34;
	private static final long P35 = 1L << 35;
	private static final long P41 = 1L << 41;
	private static final long P42 = 1L << 42;
	private static final long P48 = 1L << 48;
	private static final long P49 = 1L << 49;
	private static final long P55 = 1L << 55;
	private static final long P56 = 1L << 56;
	
	private final InputStream input;
	
    public StreamingDataInput(InputStream input)
	{
		this.input = input;
    }
	
	@Override
	public boolean readBoolean()
	{
        return readUByte() != 0;
    }

	@Override
    public byte readSByte()
	{
		return (byte) read();
    }
	
	@Override
	public int readUByte()
	{
		return read();
	}

	@Override
    public short readShort()
	{
		try {
			int b0 = input.read();
			int b1 = input.read();
			return (short)((b0 << 8) | b1);
		} catch (IOException ex) {
			throw new RuntimeIOException(ex);
		}
    }
	
	@Override
	public int readUShort()
	{
		return readShort() & 0xFFFF;
	}

	@Override
    public int readInt()
	{
		try {
			int b0 = input.read();
			int b1 = input.read();
			int b2 = input.read();
			int b3 = input.read();
			return (b0 << 24) | (b1 << 16) | (b2 << 8) | b3;
		} catch (IOException ex) {
			throw new RuntimeIOException(ex);
		}
    }
	
	@Override
	public int readUInt()
	{
		return readInt();
	}

	@Override
    public long readLong()
	{
        long i0 = readInt() & 0xFFFFFFFFL;
        long i1 = readInt() & 0xFFFFFFFFL;
        return (i0 << 32) | i1;
    }
	
	@Override
	public long readULong()
	{
		return readLong();
	}

	@Override
	public int readVarInt()
	{
		try {
			int value = input.read();
			if ((value & P7) == 0) return value - P6;
			value = (value & (P7 - 1)) | input.read() << 7;
			if ((value & P14) == 0) return value - P13;
			value = (value & (P14 - 1)) | input.read() << 14;
			if ((value & P21) == 0) return value - P20;
			value = (value & (P21 - 1)) | input.read() << 21;
			if ((value & P28) == 0) return value - P27;
			return (value & (P28 - 1)) | input.read() << 28;
		} catch (IOException ex) {
			throw new RuntimeIOException(ex);
		}
	}
	
	@Override
	public int readVarUInt()
	{
		try {
			int value = input.read();
			if ((value & P7) == 0) return value;
			value = (value & (P7 - 1)) | input.read() << 7;
			if ((value & P14) == 0) return value;
			value = (value & (P14 - 1)) | input.read() << 14;
			if ((value & P21) == 0) return value;
			value = (value & (P21 - 1)) | input.read() << 21;
			if ((value & P28) == 0) return value;
			return (value & (P28 - 1)) | input.read() << 28;
		} catch (IOException ex) {
			throw new RuntimeIOException(ex);
		}
	}
	
	@Override
	public long readVarLong()
	{
		try {
			long value = input.read();
			if ((value & P7) == 0) return value - P6;
			value = (value & (P7 - 1)) | input.read() << 7;
			if ((value & P14) == 0) return value - P13;
			value = (value & (P14 - 1)) | input.read() << 14;
			if ((value & P21) == 0) return value - P20;
			value = (value & (P21 - 1)) | input.read() << 21;
			if ((value & P28) == 0) return value - P27;
			value = (value & (P28 - 1)) | ((long)input.read()) << 28;
			if ((value & P35) == 0) return value - P34;
			value = (value & (P35 - 1)) | ((long) input.read()) << 35;
			if ((value & P42) == 0) return value - P41;
			value = (value & (P42 - 1)) | ((long) input.read()) << 42;
			if ((value & P49) == 0) return value - P48;
			value = (value & (P49 - 1)) | ((long) input.read()) << 49;
			if ((value & P56) == 0) return value - P55;
			return (value & (P56 - 1)) | ((long) input.read()) << 56;
		} catch (IOException ex) {
			throw new RuntimeIOException(ex);
		}
	}
	
	@Override
	public long readVarULong()
	{
		try {
			long value = input.read();
			if ((value & P7) == 0) return value;
			value = (value & (P7 - 1)) | input.read() << 7;
			if ((value & P14) == 0) return value;
			value = (value & (P14 - 1)) | input.read() << 14;
			if ((value & P21) == 0) return value;
			value = (value & (P21 - 1)) | input.read() << 21;
			if ((value & P28) == 0) return value;
			value = (value & (P28 - 1)) | ((long) input.read()) << 28;
			if ((value & P35) == 0) return value;
			value = (value & (P35 - 1)) | ((long) input.read()) << 35;
			if ((value & P42) == 0) return value;
			value = (value & (P42 - 1)) | ((long) input.read()) << 42;
			if ((value & P49) == 0) return value;
			value = (value & (P49 - 1)) | ((long) input.read()) << 49;
			if ((value & P56) == 0) return value;
			return (value & (P56 - 1)) | ((long) input.read()) << 56;
		} catch (IOException ex) {
			throw new RuntimeIOException(ex);
		}
	}
	
	@Override
    public float readFloat()
	{
        return Float.intBitsToFloat(readInt());
    }
	
	@Override
    public double readDouble()
	{
        return Double.longBitsToDouble(readLong());
    }
	
	@Override
	public int readChar()
	{
		return readVarUInt();
	}
	
	@Override
	public String readString()
	{
		int size = readVarUInt();
        return new String(readRawBytes(size), StandardCharsets.UTF_8);
	}
	
	@Override
    public byte[] readBytes()
	{
        int size = readVarUInt();
        return readRawBytes(size);
    }

	@Override
    public byte[] readRawBytes(int size)
	{
		try {
			byte[] result = new byte[size];
			int readSoFar = 0;
			while (readSoFar < size) {
				readSoFar += input.read(result, readSoFar, size - readSoFar);
			}
			return result;
		} catch (IOException ex) {
			throw new RuntimeIOException(ex);
		}
    }
	
	@Override
	public boolean[] readBoolArray()
	{
		int size = readVarUInt();
		boolean[] result = new boolean[size];
		for (int i = 0; i < (size + 7) / 8; i++) {
			int bvalue = readUByte();
			int remainingBits = result.length - 8 * i;

			if (remainingBits > 0)
				result[i * 8 + 0] = (bvalue & 1) > 0;
			if (remainingBits > 1)
				result[i * 8 + 1] = (bvalue & 2) > 0;
			if (remainingBits > 2)
				result[i * 8 + 2] = (bvalue & 4) > 0;
			if (remainingBits > 3)
				result[i * 8 + 3] = (bvalue & 8) > 0;
			if (remainingBits > 4)
				result[i * 8 + 4] = (bvalue & 16) > 0;
			if (remainingBits > 5)
				result[i * 8 + 5] = (bvalue & 32) > 0;
			if (remainingBits > 6)
				result[i * 8 + 6] = (bvalue & 64) > 0;
			if (remainingBits > 7)
				result[i * 8 + 7] = (bvalue & 128) > 0;
		}
		return result;
	}
	
	@Override
	public byte[] readByteArray()
	{
		return readBytes();
	}
	
	@Override
	public byte[] readUByteArray()
	{
		return readBytes();
	}
	
	@Override
	public short[] readShortArray()
	{
		return readShortArray(readVarUInt());
	}
	
	@Override
	public short[] readShortArray(int length)
	{
		short[] result = new short[length];
		for (int i = 0; i < result.length; i++)
			result[i] = readShort();
		return result;
	}
	
	@Override
	public short[] readUShortArray()
	{
		return readShortArray();
	}
	
	@Override
	public short[] readUShortArray(int length)
	{
		return readShortArray(length);
	}
	
	@Override
	public int[] readVarIntArray()
	{
		return readVarIntArray(readVarUInt());
	}
	
	@Override
	public int[] readVarIntArray(int length)
	{
		int[] result = new int[length];
		for (int i = 0; i < result.length; i++)
			result[i] = readVarInt();
		return result;
	}
	
	@Override
	public int[] readVarUIntArray()
	{
		return readVarUIntArray(readVarUInt());
	}
	
	@Override
	public int[] readVarUIntArray(int length)
	{
		int[] result = new int[length];
		for (int i = 0; i < result.length; i++)
			result[i] = readVarUInt();
		return result;
	}
	
	@Override
	public int[] readIntArray()
	{
		return readIntArray(readVarUInt());
	}
	
	@Override
	public int[] readIntArray(int length)
	{
		int[] result = new int[length];
		for (int i = 0; i < result.length; i++)
			result[i] = readInt();
		return result;
	}
	
	@Override
	public int[] readUIntArray()
	{
		int[] result = new int[readVarUInt()];
		for (int i = 0; i < result.length; i++)
			result[i] = readUInt();
		return result;
	}
	
	@Override
	public long[] readVarLongArray()
	{
		return readVarLongArray(readVarUInt());
	}
	
	@Override
	public long[] readVarLongArray(int length)
	{
		long[] result = new long[length];
		for (int i = 0; i < result.length; i++)
			result[i] = readVarLong();
		return result;
	}
	
	@Override
	public long[] readVarULongArray()
	{
		return readVarULongArray(readVarUInt());
	}
	
	@Override
	public long[] readVarULongArray(int length)
	{
		long[] result = new long[length];
		for (int i = 0; i < result.length; i++)
			result[i] = readVarULong();
		return result;
	}
	
	@Override
	public long[] readLongArray()
	{
		return readLongArray(readVarUInt());
	}
	
	@Override
	public long[] readLongArray(int length)
	{
		long[] result = new long[length];
		for (int i = 0; i < result.length; i++)
			result[i] = readLong();
		return result;
	}
	
	@Override
	public long[] readULongArray()
	{
		return readLongArray();
	}
	
	@Override
	public long[] readULongArray(int length)
	{
		return readLongArray(length);
	}
	
	@Override
	public float[] readFloatArray()
	{
		return readFloatArray(readVarUInt());
	}
	
	@Override
	public float[] readFloatArray(int length)
	{
		float[] result = new float[length];
		for (int i = 0; i < result.length; i++)
			result[i] = readFloat();
		return result;
	}
	
	@Override
	public double[] readDoubleArray()
	{
		return readDoubleArray(readVarUInt());
	}
	
	@Override
	public double[] readDoubleArray(int length)
	{
		double[] result = new double[length];
		for (int i = 0; i < result.length; i++)
			result[i] = readDouble();
		return result;
	}
	
	@Override
	public String[] readStringArray()
	{
		return readStringArray(readVarUInt());
	}
	
	@Override
	public String[] readStringArray(int length)
	{
		String[] result = new String[length];
		for (int i = 0; i < result.length; i++)
			result[i] = readString();
		return result;
	}
	
	@Override
	public LocalDate readDate()
	{
		int value = readVarInt();
		if (value == -32) // fits in a single byte and equals 31/11/1999, which is an impossible date
			return null;
		
		int year = value / (12 * 31) + 2000;
		int monthsAndDays = value % (12 * 31);
		
		if (value < 0) {
			year--;
			monthsAndDays += 12 * 31;
		}
		
		int month = monthsAndDays / 31;
		int day = monthsAndDays % 31;
		return LocalDate.of(year, month + 1, day + 1);
	}
	
	@Override
	public UUID readUUID()
	{
		long msb = readLong();
		long lsb = readLong();
		return new UUID(msb, lsb);
	}
	
	@Override
	public void skip(int bytes)
	{
		try {
			input.skip(bytes);
		} catch (IOException ex) {
			throw new RuntimeIOException(ex);
		}
	}
	
	@Override
	public boolean hasMore()
	{
		try {
			return input.available() > 0;
		} catch (IOException ex) {
			throw new RuntimeIOException(ex);
		}
	}

	@Override
	public void close()
	{
		try {
			input.close();
		} catch (IOException ex) {
			throw new RuntimeIOException(ex);
		}
	}
	
	private int read() {
		try {
			return input.read();
		} catch (IOException ex) {
			throw new RuntimeIOException(ex);
		}
	}
}
