/* MIT licensed - see LICENSE in the project root directory. */
package org.openzen.packetstreams.crypto;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import org.openzen.packetstreams.io.BytesDataInput;
import org.openzen.packetstreams.io.BytesDataOutput;

/**
 * Contains a certificate chain. A certificate chain consists of a signing root,
 * zero or more certificate chain nodes, and the signature for the final domain
 * name.
 * 
 * Certificates are only valid for a single fully-qualified domain name - 
 * wildcard certificates do not exist. (however, intermediate certificate nodes
 * can be used to implement a system with wildcard certificates)
 * 
 * The root key is used to certify the first node in the path (or the final
 * signature, if there are no nodes in the path). Each node then certifies the
 * next node. The order of nodes in path is important: each node must certify
 * the next.
 */
public class CertificateChain {
	public final CryptoVerifyKey rootKey;
	public final List<CertificateChainNode> path;
	public final byte[] signature;
	
	public CertificateChain(CryptoSigningKey rootSigningKey, CryptoPublicKey publicKey, String hostname) {
		this.rootKey = rootSigningKey.getVerifyKey();
		path = Collections.emptyList();
		
		BytesDataOutput output = new BytesDataOutput();
		output.writeString(hostname);
		output.writeRawBytes(publicKey.encode());
		signature = rootSigningKey.sign(output.toByteArray());
	}
	
	public CertificateChain(CryptoVerifyKey rootKey, List<CertificateChainNode> path, byte[] signature) {
		this.rootKey = rootKey;
		this.path = path;
		this.signature = signature;
	}
	
	public CertificateChain(CryptoProvider crypto, BytesDataInput input) {
		rootKey = crypto.decodeVerifyKey(input.readRawBytes(CryptoProvider.VERIFY_KEY_LENGTH));
		int pathLength = input.readVarUInt();
		path = new ArrayList<>();
		for (int i = 0; i < pathLength; i++) {
			String domain = input.readString();
			CryptoVerifyKey verifyKey = crypto.decodeVerifyKey(input.readRawBytes(CryptoProvider.VERIFY_KEY_LENGTH));
			long validFrom = input.readVarULong();
			long validDuration = input.readVarULong();
			byte[] signature = input.readRawBytes(CryptoProvider.SIGNATURE_LENGTH);
			path.add(new CertificateChainNode(domain, verifyKey, validFrom, validDuration, signature));
		}
		signature = input.readRawBytes(CryptoProvider.SIGNATURE_LENGTH);
	}
	
	public boolean validate(String hostname, CryptoPublicKey serverPublicKey) {
		CertificateChainNode previous = new CertificateChainNode("***", rootKey, Long.MIN_VALUE, -1, new byte[0]);
		for (CertificateChainNode node : path) {
			if (!node.validate(previous))
				return false;
			
			previous = node;
		}
		
		if (!previous.isValid(hostname, System.currentTimeMillis()))
			return false;
		
		BytesDataOutput output = new BytesDataOutput();
		output.writeString(hostname);
		output.writeRawBytes(serverPublicKey.encode());
		return previous.verifyKey.verify(output.toByteArray(), signature);
	}
	
	public void serialize(BytesDataOutput output) {
		output.writeRawBytes(rootKey.encode());
		output.writeVarUInt(path.size());
		for (CertificateChainNode node : path) {
			output.writeString(node.hostname);
			output.writeRawBytes(node.verifyKey.encode());
			output.writeVarULong(node.validFrom);
			output.writeVarULong(node.validDuration);
			output.writeRawBytes(node.signature);
		}
		output.writeRawBytes(signature);
	}
}
