/* MIT licensed - see LICENSE in the project root directory. */
package org.openzen.packetstreams.qpsp;

import org.openzen.packetstreams.Server;
import java.net.InetAddress;
import java.net.UnknownHostException;
import org.openzen.packetstreams.SigningRootValidator;
import org.openzen.packetstreams.crypto.CryptoDecryptionException;
import org.openzen.packetstreams.crypto.CryptoKeyPair;
import org.openzen.packetstreams.crypto.CryptoPublicKey;
import org.openzen.packetstreams.crypto.CryptoSharedKey;
import org.openzen.packetstreams.crypto.CryptoVerifyKey;
import org.openzen.packetstreams.crypto.CryptoProvider;
import org.openzen.packetstreams.ConnectionListener;

/**
 * Represents a client-initiating session. Only used during initialization.
 */
public class QPSPConnectionRequest {
	private final QPSPEndpoint endpoint;
	private final CryptoKeyPair keyPair;
	private final long clientNonce;
	public final String host;
	private final Server server;
	private final SigningRootValidator rootValidator;
	private final int keepaliveInterval;
	private final int maxKeepaliveInterval;
	private final long connectionId;
	private final ConnectionListener listener;
	
	private long serverNonce;
	private CryptoPublicKey serverKey;
	
	public QPSPConnectionRequest(
			QPSPEndpoint endpoint,
			Server local,
			String host,
			CryptoKeyPair keyPair,
			long clientNonce,
			long connectionId,
			SigningRootValidator rootValidator,
			ConnectionListener listener,
			int keepaliveInterval,
			int maxKeepaliveInterval) throws UnknownHostException {
		this.host = host;
		this.endpoint = endpoint;
		this.keyPair = keyPair;
		this.clientNonce = clientNonce;
		this.server = local;
		
		this.rootValidator = rootValidator;
		this.listener = listener;
		this.keepaliveInterval = keepaliveInterval;
		this.maxKeepaliveInterval = maxKeepaliveInterval;
		this.connectionId = connectionId;
	}
	
	public void preInit(long serverNonce, CryptoPublicKey publicKey) {
		this.serverNonce = serverNonce;
		this.serverKey = publicKey;
	}
	
	public boolean isValidSigningRoot(CryptoVerifyKey rootKey) {
		return rootValidator.isValid(rootKey);
	}
	
	public QPSPConnection init(
			InetAddress address,
			int port,
			long remoteConnectionID,
			int maxPacketSize,
			int maxBufferSize) {
		QPSPConnection result = new QPSPConnection(
				endpoint,
				address, port,
				connectionId,
				remoteConnectionID,
				maxPacketSize,
				maxBufferSize,
				serverNonce,
				clientNonce,
				server,
				false,
				serverKey,
				keyPair);
		result.initClient(keepaliveInterval, maxKeepaliveInterval);
		
		listener.onConnected(result);
		return result;
	}
	
	public byte[] decryptInit(byte[] init) throws CryptoDecryptionException {
		byte[] nonce = new byte[CryptoProvider.NONCE_BYTES];
		QPSPConnection.setLong(nonce, 0, clientNonce);
		QPSPConnection.setLong(nonce, 8, 1);
		
		CryptoSharedKey sharedKey = endpoint.crypto.createSharedKey(serverKey, keyPair.privateKey);
		return sharedKey.decrypt(nonce, init);
	}
}
