/* MIT licensed - see LICENSE in the project root directory. */
package org.openzen.packetstreams.qpsp.frames;

import org.openzen.packetstreams.NetworkLogger;
import org.openzen.packetstreams.io.BytesDataInput;
import org.openzen.packetstreams.io.BytesDataOutput;
import org.openzen.packetstreams.qpsp.Constants;
import org.openzen.packetstreams.qpsp.FrameData;
import org.openzen.packetstreams.qpsp.QPSPConnection;
import org.openzen.packetstreams.qpsp.QPSPStream;

public class FragEndFrame implements Frame, OutgoingFrame {
	public static FragEndFrame deserialize(BytesDataInput input, QPSPConnection connection) {
		QPSPStream stream = connection.getStream(input.readVarUInt());
		long dataseq = stream.decodeCompactedSEQ(input);
		connection.logger.log(NetworkLogger.CATEGORY_FRAMES, connection.localID, stream.id, "<- FRAGEND");
		byte[] fragment = input.readByteArray();
		return new FragEndFrame(stream, dataseq, fragment);
	}
	
	private final QPSPStream stream;
	private final long seq;
	private final byte[] data;
	
	public FragEndFrame(QPSPStream stream, long seq, byte[] data) {
		this.stream = stream;
		this.seq = seq;
		this.data = data;
	}

	@Override
	public boolean tryExecute() {
		if (!stream.hasReached(seq))
			return false;
		
		stream.finishFragmented(seq, data);
		return true;
	}
	
	@Override
	public QPSPStream getStream() {
		return stream;
	}
	
	@Override
	public long getBlockingSeq() {
		return seq;
	}

	@Override
	public int length() {
		return 1
				+ BytesDataInput.getVarUIntLength(stream.id)
				+ stream.getCompactedSEQLength(seq)
				+ BytesDataInput.getByteArrayLength(data);
	}

	@Override
	public FrameData encode() {
		BytesDataOutput output = new BytesDataOutput();
		output.writeUByte(Constants.PACKET_FRAGEND);
		output.writeVarUInt(stream.id);
		stream.encodeCompactedSEQ(output, seq);
		output.writeBytes(data);
		
		stream.logger.log(NetworkLogger.CATEGORY_FRAMES, stream.connection.localID, stream.id, "-> FRAGPART " + seq);
		return new FrameData(stream.getPriority(), seq, output.toByteArray(), false, false);
	}

	@Override
	public FrameData encodeAsFragmented() {
		throw new UnsupportedOperationException("Refragmenting of FRAGEND is not permitted");
	}
}
