/* MIT licensed - see LICENSE in the project root directory. */
package org.openzen.packetstreams.qpsp.frames;

import org.openzen.packetstreams.NetworkLogger;
import org.openzen.packetstreams.Service;
import org.openzen.packetstreams.io.BytesDataInput;
import org.openzen.packetstreams.io.BytesDataOutput;
import org.openzen.packetstreams.qpsp.Constants;
import org.openzen.packetstreams.qpsp.FrameData;
import org.openzen.packetstreams.qpsp.QPSPConnection;
import org.openzen.packetstreams.qpsp.QPSPStream;

public class StartFrame implements Frame, OutgoingFrame {
	public static StartFrame deserialize(BytesDataInput input, QPSPConnection connection) {
		QPSPStream stream = connection.getStream(input.readVarUInt());
		connection.logger.log(NetworkLogger.CATEGORY_FRAMES, connection.localID, stream.id, "<- START");
		int checksum = input.readUInt();
		int priority = input.readVarInt();
		byte[] initData = input.readByteArray();
		return new StartFrame(stream, checksum, priority, initData);
	}
	
	private final QPSPStream stream;
	private final int checksum;
	private final int priority;
	private final byte[] initData;
	
	public StartFrame(QPSPStream stream, int checksum, int priority, byte[] initData) {
		this.stream = stream;
		this.checksum = checksum;
		this.priority = priority;
		this.initData = initData;
	}

	@Override
	public boolean tryExecute() {
		Service service = stream.getService();
		if (service == null)
			return false; // OPEN or QUICKOPEN not yet received
		
		int calculatedChecksum = service.getMeta().checksum();
		if (checksum != calculatedChecksum) {
			stream.transmitServiceMeta();
		} else {
			stream.start(priority, initData);
		}
		return true;
	}
	
	@Override
	public QPSPStream getStream() {
		return stream;
	}

	@Override
	public int length() {
		return 5
				+ BytesDataInput.getVarUIntLength(stream.id)
				+ BytesDataInput.getVarIntLength(priority)
				+ BytesDataInput.getByteArrayLength(initData);
	}

	@Override
	public FrameData encode() {
		BytesDataOutput frame = new BytesDataOutput();
		frame.writeUByte(Constants.PACKET_START);
		frame.writeVarUInt(stream.id);
		frame.writeUInt(checksum);
		frame.writeVarInt(priority);
		frame.writeByteArray(initData);
		return new FrameData(priority, -1, frame.toByteArray(), false, false);
	}

	@Override
	public FrameData encodeAsFragmented() {
		BytesDataOutput frame = new BytesDataOutput();
		frame.writeUByte(Constants.PACKET_START);
		frame.writeUInt(checksum);
		frame.writeVarInt(priority);
		frame.writeByteArray(initData);
		return new FrameData(priority, -1, frame.toByteArray(), false, false);
	}
}
