/* MIT licensed - see LICENSE in the project root directory. */
package org.openzen.packetstreams.qpsp.frames;

import java.util.UUID;
import org.openzen.packetstreams.NetworkLogger;
import org.openzen.packetstreams.ServiceMeta;
import org.openzen.packetstreams.io.BytesDataInput;
import org.openzen.packetstreams.io.BytesDataOutput;
import org.openzen.packetstreams.qpsp.Constants;
import org.openzen.packetstreams.qpsp.FrameData;
import org.openzen.packetstreams.qpsp.QPSPConnection;
import org.openzen.packetstreams.qpsp.QPSPStream;

public class ServiceInfoFrame implements Frame, OutgoingFrame {
	public static ServiceInfoFrame deserialize(BytesDataInput input, QPSPConnection connection) {
		QPSPStream stream = connection.getStream(input.readVarUInt());
		UUID uuid = input.readUUID();
		int flags = input.readVarUInt();
		int priority = input.readVarInt();
		byte[] serviceInfo = input.readByteArray();
		connection.logger.log(NetworkLogger.CATEGORY_FRAMES, connection.localID, stream.id, "<- SERVICEINFO");
		return new ServiceInfoFrame(stream, new ServiceMeta(uuid, flags, priority, serviceInfo));
	}
	
	private final QPSPStream stream;
	private final ServiceMeta meta;
	
	public ServiceInfoFrame(QPSPStream stream, ServiceMeta meta) {
		this.stream = stream;
		this.meta = meta;
	}

	@Override
	public boolean tryExecute() {
		stream.deliverServiceInfo(meta);
		return true;
	}
	
	@Override
	public QPSPStream getStream() {
		return stream;
	}

	@Override
	public int length() {
		return 18
				+ BytesDataInput.getVarUIntLength(stream.id)
				+ BytesDataInput.getVarUIntLength(meta.defaultPriority)
				+ BytesDataInput.getByteArrayLength(meta.serviceInfo);
	}

	@Override
	public FrameData encode() {
		BytesDataOutput output = new BytesDataOutput();
		output.writeUByte(Constants.PACKET_SERVICEINFO);
		output.writeVarUInt(stream.id);
		output.writeUUID(meta.uuid);
		output.writeVarUInt(meta.flags);
		output.writeVarInt(meta.defaultPriority);
		output.writeByteArray(meta.serviceInfo);
		return new FrameData(stream.getPriority(), -1, output.toByteArray(), false, false);
	}

	@Override
	public FrameData encodeAsFragmented() {
		BytesDataOutput output = new BytesDataOutput();
		output.writeUByte(Constants.PACKET_SERVICEINFO);
		output.writeUUID(meta.uuid);
		output.writeVarUInt(meta.flags);
		output.writeVarInt(meta.defaultPriority);
		output.writeByteArray(meta.serviceInfo);
		return new FrameData(stream.getPriority(), -1, output.toByteArray(), false, false);
	}
}
