/* MIT licensed - see LICENSE in the project root directory. */
package org.openzen.packetstreams.qpsp;

/**
 * The PacketScheduler transmits and retransmits packets. Different interface 
 * implementations can provide different algorithms.
 */
public interface PacketScheduler {
	/**
	 * Pauses the stream. Stops it from transmitting anything.
	 */
	void pause();
	
	/**
	 * Resumes the stream after pausing. Continues data transmission.
	 */
	void resume();
	
	/**
	 * Indicates that new data is available for transmission.
	 */
	void resumeStreams();
	
	/**
	 * Informs that a packet - any packet - has been received from the remote
	 * peer.
	 */
	void onPacketReceived();
	
	/**
	 * Registers the acknowledgement of a packet. Indicates that the packet has
	 * been successfully received (and doesn't need to be retransmitted)
	 * 
	 * @param fromSeq from SEQ number
	 * @param toSeq to SEQ number
	 */
	void onAcknowledged(long fromSeq, long toSeq, NackRange[] nacks);
	
	/**
	 * Notifies the closing of a stream. All data for that stream should be
	 * cleaned up.
	 * 
	 * @param streamID 
	 */
	void onStreamClosed(int streamID);
	
	/**
	 * Called when a session has been closed. Should clean up all resources.
	 */
	void onConnectionClosed();
	
	/**
	 * Estimates the current round-trip time.
	 * 
	 * @return estimated RTT in milliseconds
	 */
	int getEstimatedRTTInMillis();
	
	/**
	 * Returns the timestamp of the moment when the last packet has been sent.
	 * Used by the keepalive system to make sure a packet is sent at least
	 * sometimes.
	 * 
	 * @return timestamp of last transmitted timestamp
	 */
	long getLastSentPacketTimestamp();
}
