/* MIT licensed - see LICENSE in the project root directory. */
package org.openzen.packetstreams.qpsp.frames;

import org.openzen.packetstreams.NetworkLogger;
import org.openzen.packetstreams.io.BytesDataInput;
import org.openzen.packetstreams.io.BytesDataOutput;
import org.openzen.packetstreams.qpsp.Constants;
import org.openzen.packetstreams.qpsp.FrameData;
import org.openzen.packetstreams.qpsp.NackRange;
import org.openzen.packetstreams.qpsp.QPSPConnection;
import org.openzen.packetstreams.qpsp.QPSPStream;

public class AckFrame implements Frame, OutgoingFrame {
	public static AckFrame deserialize(BytesDataInput input, QPSPConnection connection) {
		long ackseq = connection.decodeCompactedSEQ(input);
		NackRange[] nacks = new NackRange[input.readVarUInt()];
		StringBuilder nacksString = new StringBuilder();
		long last = ackseq;
		for (int i = 0; i < nacks.length; i++) {
			long delta = input.readVarULong();
			long nackseq = last - (delta >> 1);
			int length;
			if ((delta & 1) == 1)
				length = input.readVarUInt();
			else
				length = 1;

			nacks[i] = new NackRange(nackseq, length);
			last = nackseq;

			if (i > 0) nacksString.append(", ");
			if (length > 1)
				nacksString.append(nackseq).append("+").append(length);
			else
				nacksString.append(nackseq);
		}
		long startOfRange = last - input.readVarULong();

		if (nacksString.length() > 0)
			connection.logger.log(NetworkLogger.CATEGORY_FRAMES, connection.localID, -1, "<- ACK " + startOfRange + "-" + ackseq + " NACK " + nacksString);
		else
			connection.logger.log(NetworkLogger.CATEGORY_FRAMES, connection.localID, -1, "<- ACK " + startOfRange + "-" + ackseq);

		return new AckFrame(connection, startOfRange, ackseq, nacks);
	}
	
	private final QPSPConnection connection;
	private final long from;
	private final long to;
	private final NackRange[] nacks;
	
	public AckFrame(QPSPConnection connection, long from, long to, NackRange[] nacks) {
		this.connection = connection;
		this.from = from;
		this.to = to;
		this.nacks = nacks;
	}

	@Override
	public boolean tryExecute() {
		connection.scheduler.onAcknowledged(from, to, nacks);
		return true;
	}

	@Override
	public QPSPStream getStream() {
		return null;
	}

	@Override
	public int length() {
		// a bit hard to calculate...
		return encode().data.length;
	}

	@Override
	public FrameData encode() {
		BytesDataOutput output = new BytesDataOutput();
		output.writeUByte(Constants.PACKET_ACK);
		connection.encodeCompactedSEQ(output, to);
		output.writeVarUInt(nacks.length);
		long last = to;
		for (NackRange range : nacks) {
			long delta = last - range.seq;
			if (delta < 0)
				throw new AssertionError();
			if (range.length == 1) {
				output.writeVarULong(delta * 2);
			} else {
				output.writeVarULong(delta * 2 + 1);
				output.writeVarUInt(range.length);
			}
			last = range.seq;
		}
		output.writeVarULong(last - from);
		return new FrameData(Integer.MAX_VALUE, -1, output.toByteArray(), true, false);
	}

	@Override
	public FrameData encodeAsFragmented() {
		return encode();
	}
}
