/* MIT licensed - see LICENSE in the project root directory. */
package org.openzen.packetstreams;

import org.openzen.packetstreams.crypto.CertificateChain;
import org.openzen.packetstreams.crypto.CryptoKeyPair;

/**
 * Represents a single server on the system. A server may expose any number of
 * services, each on a unique path.
 */
public interface Server {
	/**
	 * Resolves the service at the given path. If there is no service on the
	 * given path, should return null.
	 * 
	 * @param path service path
	 * @return the service, or null if there is none here
	 */
	Service getService(String path);
	
	/**
	 * Retrieves the key pair for this server. Should return the same key pair
	 * with every invocation; since the certificate depends on it.
	 * 
	 * @return server key pair
	 */
	CryptoKeyPair getKeyPair();
	
	/**
	 * Retrieves the certificate for this server. If this is a client-side
	 * server, this method doesn't need to be implemented and may throw an
	 * exception if invoked.
	 * 
	 * @return server certificate
	 */
	CertificateChain getCertificate();
	
	/**
	 * Returns the maximum number of streams in a connection.
	 * 
	 * @return 
	 */
	default int getMaxStreams() {
		return 16384;
	}
	
	/**
	 * Returns the maximum size of a packet. Clients must not generate packets
	 * larger than this maximum. This helps for resource-constrained devices.
	 * 
	 * @return 
	 */
	default int getMaxPacketSize() {
		return 65536;
	}
	
	/**
	 * Returns the maximum size of all unfinished packets. Unfinished packets
	 * are packets that couldn't yet be delivered to the application.
	 * Resource-constrained devices can set this value to a lower value.
	 * 
	 * @return maximum buffer size
	 */
	default int getMaxBufferSize() {
		return 64 * getMaxPacketSize();
	}
}
