/* MIT licensed - see LICENSE in the project root directory. */
package org.openzen.packetstreams.crypto;

import org.openzen.packetstreams.io.BytesDataInput;
import org.openzen.packetstreams.io.BytesDataOutput;

/**
 * Contains a single node in a certificate chain.
 */
public class CertificateChainNode {
	public final String hostname;
	public final CryptoVerifyKey verifyKey;
	public final long validFrom;
	public final long validDuration;
	public final byte[] signature;
	
	public CertificateChainNode(
			String hostname,
			CryptoVerifyKey verifyKey,
			long validFrom,
			long validDuration,
			byte[] signature) {
		this.hostname = hostname;
		this.verifyKey = verifyKey;
		this.validFrom = validFrom;
		this.validDuration = validDuration;
		this.signature = signature;
	}
	
	public CertificateChainNode(CryptoProvider crypto, BytesDataInput input) {
		hostname = input.readString();
		verifyKey = crypto.decodeVerifyKey(input.readRawBytes(CryptoProvider.VERIFY_KEY_LENGTH));
		validFrom = input.readVarULong();
		validDuration = input.readVarULong();
		signature = input.readRawBytes(CryptoProvider.SIGNATURE_LENGTH);
	}
	
	public boolean validate(CertificateChainNode previous) {
		if (!previous.authorizes(hostname))
			return false;
		if (validFrom < previous.validFrom || (validFrom + validDuration) > (previous.validFrom + previous.validDuration))
			return false;
		
		BytesDataOutput output = new BytesDataOutput();
		output.writeString(hostname);
		output.writeRawBytes(verifyKey.encode());
		output.writeVarULong(validFrom);
		output.writeVarULong(validDuration);
		return verifyKey.verify(output.toByteArray(), signature);
	}
	
	public boolean authorizes(String domainName) {
		if (this.hostname.equals(domainName))
			return true;
		
		if (hostname.equals("***")) // signing root
			return true;
		if (hostname.startsWith("**."))
			return domainName.endsWith(hostname.substring(2)) || domainName.equals(hostname.substring(3));
		if (hostname.startsWith("*.")) {
			if (!domainName.endsWith(hostname.substring(1)) && !domainName.equals(hostname.substring(2)))
				return false;
			if (domainName.indexOf('.') != domainName.length() - hostname.length() + 1)
				return false;
			
			return true;
		}
		
		return false;
	}
	
	public boolean isValid(String domainName, long time) {
		return authorizes(domainName)
				&& time >= validFrom
				&& time < (validFrom + validDuration);
	}
}
