/* MIT licensed - see LICENSE in the project root directory. */
package org.openzen.packetstreams.qpsp;

import java.util.Arrays;
import java.util.UUID;
import org.openzen.packetstreams.BasePacketStream;
import org.openzen.packetstreams.NetworkLogger;
import org.openzen.packetstreams.PacketHints;
import org.openzen.packetstreams.Server;
import org.openzen.packetstreams.Service;
import org.openzen.packetstreams.ServiceConnector;
import org.openzen.packetstreams.ServiceMeta;
import org.openzen.packetstreams.io.BytesDataInput;
import org.openzen.packetstreams.io.BytesDataOutput;
import org.openzen.packetstreams.qpsp.frames.StartFrame;
import org.openzen.packetstreams.ServiceStream;
import org.openzen.packetstreams.crypto.CryptoProvider;
import org.openzen.packetstreams.qpsp.frames.CloseFrame;
import org.openzen.packetstreams.qpsp.frames.DataFrame;
import org.openzen.packetstreams.qpsp.frames.FragEndFrame;
import org.openzen.packetstreams.qpsp.frames.FragPartFrame;
import org.openzen.packetstreams.qpsp.frames.FragStartFrame;
import org.openzen.packetstreams.qpsp.frames.OpenFrame;
import org.openzen.packetstreams.qpsp.frames.OutgoingFrame;
import org.openzen.packetstreams.qpsp.frames.ServiceInfoFrame;

/**
 * Represents a single stream.
 */
public class QPSPStream extends BasePacketStream implements StreamMultiplexer.Stream {
	public final QPSPConnection connection;
	public final NetworkLogger logger;
	public final int id;
	
	private long outgoingSeq = 0;
	private long currentSeq = 0;
	private int priority = 0;
	
	private FrameData outgoingFragmenting = null;
	private int outgoingFragmentingOffset = 0;
	
	private Service service;
	private final ServiceConnector connector;
	private ServiceStream serviceStream;
	
	private BytesDataOutput incomingFragment = null;
	
	private boolean closed = false;
	private boolean connectedWithUpdatedInfo = false;
	
	public QPSPStream(QPSPConnection connection, int id, ServiceConnector connector) {
		this.connection = connection;
		logger = connection.logger;
		this.id = id;
		this.connector = connector;
	}
	
	public void open(String path, boolean quick) {
		connection.send(new OpenFrame(this, path, !quick).encode(), true);
	}
	
	public void connect(ServiceMeta meta) {
		if (connector == null)
			throw new IllegalStateException("This is not a client stream");
		
		this.priority = meta.defaultPriority; // TODO: allow connectors to specify their desired priority
		
		byte[] init;
		if (serviceStream == null) {
			init = connector.connect(meta);
			
			serviceStream = connector.onConnected(this);
			serviceStream.onConnected();
		} else {
			init = connector.connectWithUpdatedMeta(meta);
			if (init == null) {
				close(Constants.CLOSE_REQUESTED_BY_APPLICATION, new byte[0]);
				return;
			}
		}
		
		connection.send(new StartFrame(this, meta.checksum(), priority, init).encode(), true);
	}
	
	public Server getServer() {
		return connection.server;
	}
	
	public Service getService() {
		return service;
	}
	
	@Override
	public int getId() {
		return id;
	}
	
	@Override
	public int getPriority() {
		return priority;
	}
	
	public long decodeCompactedSEQ(BytesDataInput input) {
		// TODO: implement correct algorithm
		return input.readVarULong();
	}
	
	public void encodeCompactedSEQ(BytesDataOutput output, long value) {
		// TODO: implement correct algorithm
		output.writeVarULong(value);
	}
	
	public int getCompactedSEQLength(long value) {
		return BytesDataInput.getVarULongLength(value);
	}
	
	@Override
	public void resume() {
		connection.runOnNetworkThread(() -> {
			connection.resume(this);
		});
	}

	@Override
	public void close(int reason, byte[] info) {
		connection.runOnNetworkThread(() -> {
			long seq = newSeq();
			connection.send(new CloseFrame(this, seq, reason, info).encode(), true);
		});
	}
	
	@Override
	public CryptoProvider getCrypto() {
		return connection.getCrypto();
	}
	
	@Override
	public int getEstimatedRTTInMillis() {
		return connection.getEstimatedRTTInMillis();
	}
	
	public void setService(Service service) {
		this.service = service;
	}
	
	public void transmitServiceMeta() {
		connection.send(new ServiceInfoFrame(this, service.getMeta()).encode(), true);
	}
	
	public void start(int priority, byte[] initData) {
		if (serviceStream != null) {
			connection.logger.log(NetworkLogger.CATEGORY_STREAMS, connection.localID, id, "Skipping double start");
			return;
		}
		
		serviceStream = service.open(this, initData);
		serviceStream.onConnected();
		this.priority = priority;
	}
	
	public boolean isStarted() {
		return serviceStream != null;
	}
	
	public void deliverServiceInfo(ServiceMeta meta) {
		if (connectedWithUpdatedInfo)
			return;
		
		connect(meta);
		connectedWithUpdatedInfo = true;
	}
	
	public void deliver(long seq, byte[] packet) {
		connection.assertOnNetworkThread();
		
		if (seq != currentSeq) {
			connection.logger.log(NetworkLogger.CATEGORY_FRAMES, connection.localID, id, "Preventing duplicate delivery of " + seq);
			return;
		}
		
		logger.log(NetworkLogger.CATEGORY_FRAMES, connection.localID, id, "<- DELIVERED " + seq);
		serviceStream.onReceived(packet);
		currentSeq++;
	}
	
	public void deliverClose(long seq, int reason, byte[] info) {
		connection.assertOnNetworkThread();
		
		if (seq != currentSeq) {
			connection.logger.log(NetworkLogger.CATEGORY_FRAMES, connection.localID, id, "Preventing duplicate delivery of " + seq);
			return;
		}
		
		if (reason != Constants.CLOSE_REQUESTED_BY_PEER)
			close(Constants.CLOSE_REQUESTED_BY_PEER, new byte[0]);
		
		service = null;
		currentSeq++;
		closed = true;
		
		connection.resume(this); // makes sure the final ack gets transmitted
		connection.onClosed(id);
		if (serviceStream != null)
			serviceStream.onConnectionClosed(reason, info);
	}
	
	public void beginFragmented(long seq, byte[] data) {
		connection.assertOnNetworkThread();
		
		if (seq != currentSeq) {
			connection.logger.log(NetworkLogger.CATEGORY_FRAMES, connection.localID, id, "Preventing duplicate delivery of " + seq);
			return;
		}
		
		if (incomingFragment != null) {
			close(Constants.CLOSE_PROTOCOL_ERROR, new byte[0]);
			return;
		}
		
		incomingFragment = new BytesDataOutput();
		incomingFragment.writeRawBytes(data);
		currentSeq++;
	}
	
	public void appendFragmented(long seq, byte[] data) {
		connection.assertOnNetworkThread();
		
		if (seq != currentSeq) {
			connection.logger.log(NetworkLogger.CATEGORY_FRAMES, connection.localID, id, "Preventing duplicate delivery of " + seq);
			return;
		}
		if (incomingFragment == null) {
			close(Constants.CLOSE_PROTOCOL_ERROR, new byte[0]);
			return;
		}
		
		incomingFragment.writeRawBytes(data);
		currentSeq++;
	}
	
	public void finishFragmented(long seq, byte[] data) {
		connection.assertOnNetworkThread();
		
		if (seq != currentSeq) {
			connection.logger.log(NetworkLogger.CATEGORY_FRAMES, connection.localID, id, "Preventing duplicate delivery of " + seq);
			return;
		}
		if (incomingFragment == null) {
			close(Constants.CLOSE_PROTOCOL_ERROR, new byte[0]);
			return;
		}
		
		incomingFragment.writeRawBytes(data);
		currentSeq++;
		
		BytesDataInput input = new BytesDataInput(incomingFragment.toByteArray());
		int fragmentType = input.readUByte();
		switch (fragmentType) {
			case Constants.PACKET_SERVICEINFO: {
				UUID uuid = input.readUUID();
				int flags = input.readUByte();
				int defaultPriority = input.readVarInt();
				byte[] serviceInfo = input.readByteArray();
				deliverServiceInfo(new ServiceMeta(uuid, flags, defaultPriority, serviceInfo));
				break;
			}
			case Constants.PACKET_START: {
				int checksum = input.readVarUInt();
				int priority = input.readVarUInt();
				byte[] initData = input.readByteArray();
				connection.offer(new StartFrame(this, checksum, priority, initData));
				break;
			}
			case Constants.PACKET_DATA: {
				byte[] packetData = input.readByteArray();
				serviceStream.onReceived(packetData);
				deliver(seq, packetData);
				break;
			}
			case Constants.PACKET_CLOSE_STREAM: {
				int reason = input.readVarUInt();
				byte[] closeData = input.readByteArray();
				deliverClose(seq, reason, closeData);
				break;
			}
			default:
				close(Constants.CLOSE_PROTOCOL_ERROR, new byte[0]);
				break;
		}
		incomingFragment = null;
	}
	
	public boolean hasReached(long seq) {
		return seq <= currentSeq;
	}
	
	@Override
	public FrameData next(int availableSpaceInPacket) {
		if (closed)
			return null;
		if (outgoingFragmenting != null)
			return nextFragment(availableSpaceInPacket).encode();
		
		OutgoingFrame next = loadNext(availableSpaceInPacket);
		if (next == null)
			return null;
		
		if (next.length() > availableSpaceInPacket) {
			outgoingFragmenting = next.encodeAsFragmented();
			outgoingFragmentingOffset = 0;
			next = nextFragment(availableSpaceInPacket);
		}
		return next.encode();
	}
	
	private OutgoingFrame loadNext(int availableSpaceInPacket) {
		connection.assertOnNetworkThread();
		
		if (serviceStream == null)
			return null;
		
		int recommendedLength = availableSpaceInPacket
                - BytesDataInput.getVarUIntLength(id)
                - BytesDataInput.getVarUIntLength(availableSpaceInPacket)
                - getCompactedSEQLength(outgoingSeq)
                - 1;
		byte[] data = serviceStream.next(new PacketHints(recommendedLength));
		if (data == null)
			return null;
		
		return new DataFrame(this, newSeq(), data);
	}
	
	private OutgoingFrame nextFragment(int size) {
		size -= 1
				- BytesDataInput.getVarUIntLength(id)
				- getCompactedSEQLength(currentSeq)
				- BytesDataInput.getVarUIntLength(size);
		
		if (size < 0)
			throw new AssertionError();
		
		int until = Math.min(outgoingFragmenting.data.length, outgoingFragmentingOffset + size);
		byte[] fragment = Arrays.copyOfRange(outgoingFragmenting.data, outgoingFragmentingOffset, until);
		if (outgoingFragmentingOffset == 0) {
			long seq = outgoingFragmenting.seq;
			outgoingFragmentingOffset += fragment.length;
			return new FragStartFrame(this, seq, fragment);
		} else if (until < outgoingFragmenting.data.length) {
			long seq = newSeq();
			outgoingFragmentingOffset += fragment.length;
			return new FragPartFrame(this, seq, fragment);
		} else {
			long seq = newSeq();
			outgoingFragmenting = null;
			return new FragEndFrame(this, seq, fragment);
		}
	}
	
	public long newSeq() {
		return outgoingSeq++;
	}
	
	public void closeNow(int reason) {
		if (serviceStream == null)
			return;
		
		// TODO: deliver propery info
		serviceStream.onConnectionClosed(reason, new byte[0]);
		serviceStream = null;
	}
}
