/* MIT licensed - see LICENSE in the project root directory. */
package org.openzen.packetstreams.qpsp.frames;

import java.nio.charset.StandardCharsets;
import org.openzen.packetstreams.NetworkLogger;
import org.openzen.packetstreams.Server;
import org.openzen.packetstreams.Service;
import org.openzen.packetstreams.io.BytesDataInput;
import org.openzen.packetstreams.io.BytesDataOutput;
import org.openzen.packetstreams.qpsp.Constants;
import org.openzen.packetstreams.qpsp.FrameData;
import org.openzen.packetstreams.qpsp.QPSPConnection;
import org.openzen.packetstreams.qpsp.QPSPStream;

public class OpenFrame implements Frame, OutgoingFrame {
	public static OpenFrame deserializeOpen(BytesDataInput input, QPSPConnection connection) {
		QPSPStream stream = connection.getStream(input.readVarUInt());
		String path = input.readString();
		connection.logger.log(NetworkLogger.CATEGORY_FRAMES, connection.localID, stream.id, "<- OPEN " + path);
		return new OpenFrame(stream, path, true);
	}
	
	public static OpenFrame deserializeQuickOpen(BytesDataInput input, QPSPConnection connection) {
		QPSPStream stream = connection.getStream(input.readVarUInt());
		String path = input.readString();
		connection.logger.log(NetworkLogger.CATEGORY_FRAMES, connection.localID, stream.id, "<- QUICKOPEN " + path);
		return new OpenFrame(stream, path, false);
	}
	
	private final QPSPStream stream;
	private final String path;
	private final boolean emitServiceInfo;
	
	public OpenFrame(QPSPStream stream, String path, boolean emitServiceInfo) {
		this.stream = stream;
		this.path = path;
		this.emitServiceInfo = emitServiceInfo;
	}
	
	@Override
	public boolean tryExecute() {
		Server server = stream.getServer();
		Service service = server.getService(path);
		if (service == null) {
			stream.close(Constants.CLOSE_SERVICE_NOT_FOUND, new byte[0]);
		} else {
			stream.setService(service);
		
			if (emitServiceInfo)
				stream.transmitServiceMeta();
		}
		
		return true;
	}
	
	@Override
	public QPSPStream getStream() {
		return stream;
	}

	@Override
	public int length() {
		return 1
				+ BytesDataInput.getVarUIntLength(stream.getId())
				+ BytesDataInput.getByteArrayLength(path.getBytes(StandardCharsets.UTF_8));
	}

	@Override
	public FrameData encode() {
		BytesDataOutput frame = new BytesDataOutput();
		frame.writeUByte(emitServiceInfo ? Constants.PACKET_OPEN : Constants.PACKET_QUICKOPEN);
		frame.writeVarUInt(stream.getId());
		frame.writeString(path);
		return new FrameData(stream.getPriority(), -1, frame.toByteArray(), false, false);
	}

	@Override
	public FrameData encodeAsFragmented() {
		throw new UnsupportedOperationException("Not supported yet.");
	}

	@Override
	public long getBlockingSeq() {
		return -1;
	}
}
