/* MIT licensed - see LICENSE in the project root directory. */
package org.openzen.packetstreams.qpsp;

import java.util.Arrays;

/**
 * Multiplexes data from multiple streams together, taking into account the
 * priority of these streams.
 */
public class StreamMultiplexer {
	private Stream[] activeStreams = new Stream[4];
	private int activeStreamCount = 0;
	
	public void onPriorityChanged(Stream stream) {
		sort();
	}
	
	public int peekPriority() {
		return activeStreamCount == 0 ? Integer.MIN_VALUE : activeStreams[0].getPriority();
	}
	
	public void resume(Stream stream) {
		int position = getInsertPosition(stream);
		if (position == -1)
			return; // already active
		
		if (activeStreamCount == activeStreams.length)
			activeStreams = Arrays.copyOf(activeStreams, activeStreams.length * 2);
		
		for (int i = activeStreamCount; i > position; i--)
			activeStreams[i] = activeStreams[i - 1];

		activeStreams[position] = stream;
		activeStreamCount++;
	}
	
	public String getActive() {
		StringBuilder active = new StringBuilder();
		for (int i = 0; i < activeStreamCount; i++) {
			if (i > 0)
				active.append(", ");
			active.append(activeStreams[i].getId());
		}
		return active.toString();
	}
	
	private int getInsertPosition(Stream stream) {
		if (activeStreamCount == 0)
			return 0;
		
		for (int i = 0; i < activeStreamCount; i++) {
			if (activeStreams[i] == stream)
				return -1;
			else if (stream.getPriority() > activeStreams[i].getPriority())
				return i;
		}
		
		return activeStreamCount;
	}
	
	public FrameData next(int availableSpaceInPacket) {
		FrameData result = null;
		int removed = 0;
		
		for (int i = 0; i < activeStreamCount; i++) {
			if (removed != 0)
				activeStreams[i - removed] = activeStreams[i];
			
			if (result == null) {
				result = activeStreams[i].next(availableSpaceInPacket);
				if (result == null)
					removed++;
			}
		}
		activeStreamCount -= removed;
		
		return result;
	}
	
	private void sort() {
		Arrays.sort(activeStreams, 0, activeStreamCount, (a, b) -> a.getPriority() - b.getPriority());
	}
	
	public static interface Stream {
		int getId();
		
		int getPriority();
		
		FrameData next(int availableSpaceInPacket);
	}
}
