/* MIT licensed - see LICENSE in the project root directory. */
package org.openzen.packetstreams.qpsp.frames;

import org.openzen.packetstreams.NetworkLogger;
import org.openzen.packetstreams.io.BytesDataInput;
import org.openzen.packetstreams.io.BytesDataOutput;
import org.openzen.packetstreams.qpsp.Constants;
import org.openzen.packetstreams.qpsp.FrameData;
import org.openzen.packetstreams.qpsp.QPSPConnection;
import org.openzen.packetstreams.qpsp.QPSPStream;

public class CloseFrame implements Frame, OutgoingFrame {
	public static CloseFrame deserialize(BytesDataInput input, QPSPConnection connection) {
		QPSPStream stream = connection.getStream(input.readVarUInt());
		long dataseq = stream.decodeCompactedSEQ(input);
		int reason = input.readVarUInt();
		connection.logger.log(NetworkLogger.CATEGORY_FRAMES, connection.localID, stream.id, "<- CLOSE " + dataseq + ": " + reason);
		byte[] closeData = input.readByteArray();
		return new CloseFrame(stream, dataseq, reason, closeData);
	}
	
	private final QPSPStream stream;
	private final long seq;
	private final int reason;
	private final byte[] info;
	
	public CloseFrame(QPSPStream stream, long seq, int reason, byte[] info) {
		this.stream = stream;
		this.seq = seq;
		this.reason = reason;
		this.info = info;
	}

	@Override
	public boolean tryExecute() {
		if (!stream.hasReached(seq))
			return false;
		
		stream.deliverClose(seq, reason, info);
		return true;
	}
	
	@Override
	public QPSPStream getStream() {
		return stream;
	}
	
	@Override
	public long getBlockingSeq() {
		return seq;
	}

	@Override
	public int length() {
		return 1
				+ BytesDataInput.getVarUIntLength(stream.id)
				+ stream.getCompactedSEQLength(seq)
				+ BytesDataInput.getVarUIntLength(reason)
				+ BytesDataInput.getVarUIntLength(info.length)
				+ info.length;
	}

	@Override
	public FrameData encode() {
		BytesDataOutput close = new BytesDataOutput();
		close.writeUByte(Constants.PACKET_CLOSE_STREAM);
		close.writeVarUInt(stream.id);
		stream.encodeCompactedSEQ(close, seq);
		close.writeVarUInt(reason);
		close.writeByteArray(info);
		stream.logger.log(NetworkLogger.CATEGORY_FRAMES, stream.connection.localID, stream.id, "-> CLOSE " + seq + ": " + reason);
		return new FrameData(stream.getPriority(), seq, close.toByteArray(), false, false);
	}

	@Override
	public FrameData encodeAsFragmented() {
		BytesDataOutput close = new BytesDataOutput();
		close.writeUByte(Constants.PACKET_CLOSE_STREAM);
		close.writeVarUInt(reason);
		close.writeByteArray(info);
		stream.logger.log(NetworkLogger.CATEGORY_FRAMES, stream.connection.localID, stream.id, "-> CLOSE (fragmented) " + seq + ": " + reason);
		return new FrameData(stream.getPriority(), seq, close.toByteArray(), false, false);
	}
}
