/* MIT licensed - see LICENSE in the project root directory. */
package org.openzen.packetstreams.qpsp;

import org.openzen.packetstreams.Server;
import java.net.InetAddress;
import org.openzen.packetstreams.SigningRootValidator;
import org.openzen.packetstreams.crypto.CryptoDecryptionException;
import org.openzen.packetstreams.crypto.CryptoKeyPair;
import org.openzen.packetstreams.crypto.CryptoPublicKey;
import org.openzen.packetstreams.crypto.CryptoSharedKey;
import org.openzen.packetstreams.crypto.CryptoVerifyKey;
import org.openzen.packetstreams.crypto.CryptoProvider;
import org.openzen.packetstreams.ConnectionListener;

/**
 * Represents a client-initiating session. Only used during initialization.
 */
public class QPSPPendingConnection {
	private final QPSPEndpoint endpoint;
	public final CryptoKeyPair keyPair;
	public final long clientNonce;
	public final String host;
	public final int port;
	private final Server server;
	private final SigningRootValidator rootValidator;
	private final int keepaliveInterval;
	private final int maxKeepaliveInterval;
	public final long connectionId;
	public final ConnectionListener listener;
	
	private long serverNonce;
	private CryptoPublicKey serverKey;
	
	public QPSPPendingConnection(
			QPSPEndpoint endpoint,
			Server local,
			String host,
			int port,
			CryptoKeyPair keyPair,
			long clientNonce,
			long connectionId,
			SigningRootValidator rootValidator,
			ConnectionListener listener,
			int keepaliveInterval,
			int maxKeepaliveInterval) {
		this.host = host;
		this.port = port;
		this.endpoint = endpoint;
		this.keyPair = keyPair;
		this.clientNonce = clientNonce;
		this.server = local;
		
		this.rootValidator = rootValidator;
		this.listener = listener;
		this.keepaliveInterval = keepaliveInterval;
		this.maxKeepaliveInterval = maxKeepaliveInterval;
		this.connectionId = connectionId;
	}
	
	public QPSPPendingConnection forReconnection(long connectionId, long clientNonce) {
		return new QPSPPendingConnection(
				endpoint,
				server,
				host, port,
				keyPair,
				clientNonce, 
				connectionId,
				rootValidator,
				listener,
				keepaliveInterval,
				maxKeepaliveInterval);
	}
	
	public void preInit(long serverNonce, CryptoPublicKey publicKey) {
		this.serverNonce = serverNonce;
		this.serverKey = publicKey;
	}
	
	public boolean isValidSigningRoot(CryptoVerifyKey rootKey) {
		return rootValidator.isValid(rootKey);
	}
	
	public QPSPConnection init(
			InetAddress address,
			int port,
			long remoteConnectionID,
			int maxPacketSize,
			int maxBufferSize) {
		QPSPConnection result = new QPSPConnection(
				endpoint,
				address, port,
				connectionId,
				remoteConnectionID,
				maxPacketSize,
				maxBufferSize,
				serverNonce,
				clientNonce,
				server,
				false,
				serverKey,
				keyPair,
				this);
		result.initClient(keepaliveInterval, maxKeepaliveInterval);
		
		listener.onConnected(result);
		return result;
	}
	
	public byte[] decryptInit(byte[] init) throws CryptoDecryptionException {
		byte[] nonce = new byte[CryptoProvider.NONCE_BYTES];
		QPSPConnection.setLong(nonce, 0, clientNonce);
		QPSPConnection.setLong(nonce, 8, 1);
		
		CryptoSharedKey sharedKey = endpoint.crypto.createSharedKey(serverKey, keyPair.privateKey);
		return sharedKey.decrypt(nonce, init);
	}
}
