/* MIT licensed - see LICENSE in the project root directory. */
package org.openzen.packetstreams.qpsp;

import java.util.TimerTask;

/**
 * Represents a packet in transmission. A single packet may be transmitted
 * multiple times (resulting in multiple QPSPTransmittingPacket) due to
 * retransmission.
 */
public class QPSPTransmittingPacket implements Comparable<QPSPTransmittingPacket> {
	public final long seq;
	public final QPSPPacket packet;
	public final long timestamp = System.currentTimeMillis();
	public boolean retransmitted = false;
	public boolean lost = false;
	public final long originalSeq;
	public final QPSPTransmittingPacket retransmissionOf;
	
	private final StandardPacketScheduler scheduler;
	private TimerTask retransmissionTimer;
	private boolean inFastRetransmit;
	
	public QPSPTransmittingPacket(long seq, QPSPPacket packet, StandardPacketScheduler scheduler, QPSPTransmittingPacket retransmissionOf) {
		this.seq = seq;
		this.packet = packet;
		this.scheduler = scheduler;
		this.originalSeq = retransmissionOf == null ? seq : retransmissionOf.originalSeq;
		this.retransmissionOf = retransmissionOf;
	}
	
	public void retransmitAt(int retransmissionTimeout) {
		retransmissionTimer = scheduler.retransmit(this, retransmissionTimeout);
	}
	
	public void ack() {
		packet.acknowledged = true;
		
		if (retransmissionTimer != null) {
			retransmissionTimer.cancel();
			retransmissionTimer = null;
		}
	}
	
	public void fastRetransmit(int timeout) {
		if (packet.acknowledged)
			return;
		if (inFastRetransmit)
			return;
		
		if (retransmissionTimer != null)
			retransmissionTimer.cancel();
		
		inFastRetransmit = true;
		retransmissionTimer = scheduler.retransmit(this, timeout);
	}

	@Override
	public int compareTo(QPSPTransmittingPacket o) {
		// TODO: is this causing issues?
		//if (packet.priority != o.packet.priority)
		//	return o.packet.priority - packet.priority;
		
		return Long.compare(seq, o.seq);
	}
}
